<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Patient;
use App\Models\Drug;
use App\Models\MedicalRecord;
use App\Models\User;
use App\Models\Prescription;
use Carbon\Carbon;

class HealthcareDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('Seeding healthcare data...');

        // Create a doctor user if it doesn't exist
        $doctor = User::firstOrCreate(
            ['email' => 'doctor@hospital.com'],
            [
                'name' => 'Dr. Sarah Johnson',
                'password' => bcrypt('password'),
                'email_verified_at' => now(),
            ]
        );

        // Seed drugs
        $this->seedDrugs();

        // Seed patients (no user accounts needed)
        $this->seedPatients($doctor->id);

        $this->command->info('Healthcare data seeded successfully!');
    }

    private function seedDrugs()
    {
        $drugs = [
            [
                'name' => 'Aspirin',
                'generic_name' => 'Acetylsalicylic Acid',
                'manufacturer' => 'Bayer',
                'category' => 'NSAID',
                'description' => 'Nonsteroidal anti-inflammatory drug used for pain relief and blood thinning',
                'dosage_form' => 'Tablet',
                'strength' => '81mg',
                'unit' => 'mg',
                'prescription_required' => false,
                'side_effects' => ['Stomach irritation', 'Bleeding risk', 'Allergic reactions'],
                'contraindications' => ['Bleeding disorders', 'Stomach ulcers', 'Allergy to aspirin'],
                'interactions' => ['Warfarin', 'Ibuprofen', 'Alcohol'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ],
            [
                'name' => 'Metformin',
                'generic_name' => 'Metformin Hydrochloride',
                'manufacturer' => 'Various',
                'category' => 'Antidiabetic',
                'description' => 'First-line medication for type 2 diabetes',
                'dosage_form' => 'Tablet',
                'strength' => '500mg',
                'unit' => 'mg',
                'prescription_required' => true,
                'side_effects' => ['Nausea', 'Diarrhea', 'Stomach upset'],
                'contraindications' => ['Kidney disease', 'Liver disease', 'Heart failure'],
                'interactions' => ['Alcohol', 'Contrast dye', 'Furosemide'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ],
            [
                'name' => 'Lisinopril',
                'generic_name' => 'Lisinopril',
                'manufacturer' => 'Various',
                'category' => 'ACE Inhibitor',
                'description' => 'Angiotensin-converting enzyme inhibitor for hypertension and heart failure',
                'dosage_form' => 'Tablet',
                'strength' => '10mg',
                'unit' => 'mg',
                'prescription_required' => true,
                'side_effects' => ['Dry cough', 'Dizziness', 'High potassium'],
                'contraindications' => ['Pregnancy', 'Kidney disease', 'High potassium'],
                'interactions' => ['Potassium supplements', 'NSAIDs', 'Lithium'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ],
            [
                'name' => 'Atorvastatin',
                'generic_name' => 'Atorvastatin Calcium',
                'manufacturer' => 'Pfizer',
                'category' => 'Statin',
                'description' => 'HMG-CoA reductase inhibitor for cholesterol management',
                'dosage_form' => 'Tablet',
                'strength' => '20mg',
                'unit' => 'mg',
                'prescription_required' => true,
                'side_effects' => ['Muscle pain', 'Liver problems', 'Memory issues'],
                'contraindications' => ['Liver disease', 'Pregnancy', 'Active liver disease'],
                'interactions' => ['Grapefruit juice', 'Warfarin', 'Digoxin'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ],
            [
                'name' => 'Insulin Glargine',
                'generic_name' => 'Insulin Glargine',
                'manufacturer' => 'Sanofi',
                'category' => 'Insulin',
                'description' => 'Long-acting insulin for diabetes management',
                'dosage_form' => 'Injection',
                'strength' => '100 units/mL',
                'unit' => 'units',
                'prescription_required' => true,
                'side_effects' => ['Low blood sugar', 'Weight gain', 'Injection site reactions'],
                'contraindications' => ['Low blood sugar', 'Allergy to insulin'],
                'interactions' => ['Alcohol', 'Beta-blockers', 'Thiazides'],
                'storage_conditions' => 'Refrigerate, do not freeze',
                'expiry_alert_days' => 7,
                'is_active' => true,
            ],
            [
                'name' => 'Warfarin',
                'generic_name' => 'Warfarin Sodium',
                'manufacturer' => 'Various',
                'category' => 'Anticoagulant',
                'description' => 'Blood thinner to prevent blood clots',
                'dosage_form' => 'Tablet',
                'strength' => '5mg',
                'unit' => 'mg',
                'prescription_required' => true,
                'side_effects' => ['Bleeding', 'Bruising', 'Nausea'],
                'contraindications' => ['Active bleeding', 'Pregnancy', 'Severe liver disease'],
                'interactions' => ['Aspirin', 'NSAIDs', 'Antibiotics'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ]
        ];

        foreach ($drugs as $drugData) {
            Drug::create($drugData);
        }

        $this->command->info('Drugs seeded successfully!');
    }

    private function seedPatients($doctorId)
    {
        $patients = [
            [
                'user_id' => null, // Patients don't have user accounts
                'patient_id' => 'P' . str_pad(1, 6, '0', STR_PAD_LEFT),
                'first_name' => 'John',
                'last_name' => 'Smith',
                'date_of_birth' => '1985-03-15',
                'gender' => 'male',
                'email' => 'john.smith@example.com',
                'phone' => '+1-555-0123',
                'address' => '123 Main St, City, State 12345',
                'emergency_contact_name' => 'Jane Smith',
                'emergency_contact_phone' => '+1-555-0124',
                'insurance_provider' => 'Blue Cross Blue Shield',
                'insurance_number' => 'BC123456789',
                'medical_history' => 'Hypertension, Diabetes Type 2',
                'allergies' => 'Penicillin, Shellfish',
                'blood_type' => 'A+',
                'height' => 178.0, // cm
                'weight' => 81.6, // kg
            ],
            [
                'user_id' => null, // Patients don't have user accounts
                'patient_id' => 'P' . str_pad(2, 6, '0', STR_PAD_LEFT),
                'first_name' => 'Sarah',
                'last_name' => 'Johnson',
                'date_of_birth' => '1990-07-22',
                'gender' => 'female',
                'email' => 'sarah.johnson@example.com',
                'phone' => '+1-555-0125',
                'address' => '456 Oak Ave, City, State 12345',
                'emergency_contact_name' => 'Mike Johnson',
                'emergency_contact_phone' => '+1-555-0126',
                'insurance_provider' => 'Aetna',
                'insurance_number' => 'AE987654321',
                'medical_history' => 'Asthma, Seasonal Allergies',
                'allergies' => 'Pollen, Dust',
                'blood_type' => 'O-',
                'height' => 165.0, // cm
                'weight' => 58.0, // kg
            ],
            [
                'user_id' => null, // Patients don't have user accounts
                'patient_id' => 'P' . str_pad(3, 6, '0', STR_PAD_LEFT),
                'first_name' => 'Michael',
                'last_name' => 'Brown',
                'date_of_birth' => '1978-11-08',
                'gender' => 'male',
                'phone' => '+1-555-0127',
                'address' => '789 Pine St, City, State 12345',
                'emergency_contact_name' => 'Lisa Brown',
                'emergency_contact_phone' => '+1-555-0128',
                'insurance_provider' => 'Cigna',
                'insurance_number' => 'CI456789123',
                'medical_history' => 'Heart Disease, High Cholesterol',
                'allergies' => 'None known',
                'blood_type' => 'B+',
                'height' => 180.0, // cm
                'weight' => 90.0, // kg
            ],
            [
                'user_id' => null, // Patients don't have user accounts
                'patient_id' => 'P' . str_pad(4, 6, '0', STR_PAD_LEFT),
                'first_name' => 'Emily',
                'last_name' => 'Davis',
                'date_of_birth' => '1995-01-14',
                'gender' => 'female',
                'email' => 'emily.davis@example.com',
                'phone' => '+1-555-0129',
                'address' => '321 Elm St, City, State 12345',
                'emergency_contact_name' => 'Robert Davis',
                'emergency_contact_phone' => '+1-555-0130',
                'insurance_provider' => 'UnitedHealth',
                'insurance_number' => 'UH789123456',
                'medical_history' => 'Migraine, Anxiety',
                'allergies' => 'Latex, Nuts',
                'blood_type' => 'AB+',
                'height' => 170.0, // cm
                'weight' => 65.0, // kg
            ],
            [
                'user_id' => null, // Patients don't have user accounts
                'patient_id' => 'P' . str_pad(5, 6, '0', STR_PAD_LEFT),
                'first_name' => 'David',
                'last_name' => 'Wilson',
                'date_of_birth' => '1982-09-30',
                'gender' => 'male',
                'email' => 'david.wilson@example.com',
                'phone' => '+1-555-0131',
                'address' => '654 Maple Dr, City, State 12345',
                'emergency_contact_name' => 'Susan Wilson',
                'emergency_contact_phone' => '+1-555-0132',
                'insurance_provider' => 'Humana',
                'insurance_number' => 'HU321654987',
                'medical_history' => 'Arthritis, Depression',
                'allergies' => 'Sulfa drugs',
                'blood_type' => 'A-',
                'height' => 175.0, // cm
                'weight' => 78.0, // kg
            ]
        ];

        foreach ($patients as $patientData) {
            $patient = Patient::create($patientData);

            // Create medical records for some patients
            if (rand(0, 1)) {
                MedicalRecord::create([
                    'patient_id' => $patient->id,
                    'doctor_id' => $doctorId,
                    'visit_date' => now()->subDays(rand(1, 30)),
                    'visit_type' => 'checkup',
                    'symptoms' => 'No acute symptoms',
                    'diagnosis' => 'Healthy',
                    'treatment_plan' => 'Continue current lifestyle',
                    'notes' => 'Patient in good health',
                    'vital_signs' => [
                        'blood_pressure' => '120/80',
                        'heart_rate' => '72',
                        'temperature' => '98.6°F',
                        'weight' => '150 lbs',
                        'height' => '5\'8"'
                    ],
                    'follow_up_date' => now()->addMonths(6),
                ]);
            }

            // Create prescriptions for some patients
            if (rand(0, 1)) {
                $drugs = Drug::inRandomOrder()->limit(rand(1, 3))->get();
                foreach ($drugs as $drug) {
                    Prescription::create([
                        'patient_id' => $patient->id,
                        'doctor_id' => $doctorId,
                        'drug_id' => $drug->id,
                        'prescription_number' => 'RX-' . rand(100000, 999999),
                        'dosage' => $drug->strength,
                        'frequency' => 'Once daily',
                        'duration' => 30,
                        'quantity' => 30,
                        'instructions' => 'Take with food',
                        'prescription_date' => now()->subDays(rand(1, 10)),
                        'valid_until' => now()->addDays(30),
                        'status' => 'active',
                        'notes' => 'Regular medication',
                        'refill_count' => 0,
                        'max_refills' => 3,
                    ]);
                }
            }
        }

        $this->command->info('Patients seeded successfully!');
    }
}
