<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('operations', function (Blueprint $table) {
            $table->string('operation_number')->unique()->after('id');
            $table->foreignId('patient_id')->constrained()->onDelete('cascade')->after('operation_number');
            $table->foreignId('visit_id')->nullable()->constrained()->onDelete('set null')->after('patient_id');
            $table->foreignId('surgeon_id')->constrained('users')->onDelete('cascade')->after('visit_id');
            $table->foreignId('assistant_surgeon_id')->nullable()->constrained('users')->onDelete('set null')->after('surgeon_id');
            $table->foreignId('anesthesiologist_id')->nullable()->constrained('users')->onDelete('set null')->after('assistant_surgeon_id');
            $table->foreignId('operation_room_id')->nullable()->constrained()->onDelete('set null')->after('anesthesiologist_id');
            $table->string('surgery_type')->after('operation_room_id');
            $table->text('procedure_name')->after('surgery_type');
            $table->text('diagnosis')->nullable()->after('procedure_name');
            $table->text('indication')->nullable()->after('diagnosis');
            $table->date('scheduled_date')->after('indication');
            $table->time('scheduled_time')->after('scheduled_date');
            $table->integer('estimated_duration')->default(60)->after('scheduled_time'); // in minutes
            $table->enum('status', ['scheduled', 'preparing', 'in_progress', 'completed', 'cancelled', 'postponed'])->default('scheduled')->after('estimated_duration');
            $table->timestamp('started_at')->nullable()->after('status');
            $table->timestamp('completed_at')->nullable()->after('started_at');
            $table->text('pre_operative_notes')->nullable()->after('completed_at');
            $table->text('operative_notes')->nullable()->after('pre_operative_notes');
            $table->text('post_operative_notes')->nullable()->after('operative_notes');
            $table->text('complications')->nullable()->after('post_operative_notes');
            $table->text('notes')->nullable()->after('complications');
            $table->foreignId('booked_by')->constrained('users')->onDelete('cascade')->after('notes');
            
            // Indexes
            $table->index('patient_id');
            $table->index('surgeon_id');
            $table->index('operation_room_id');
            $table->index('scheduled_date');
            $table->index('status');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('operations', function (Blueprint $table) {
            $table->dropForeign(['patient_id']);
            $table->dropForeign(['visit_id']);
            $table->dropForeign(['surgeon_id']);
            $table->dropForeign(['assistant_surgeon_id']);
            $table->dropForeign(['anesthesiologist_id']);
            $table->dropForeign(['operation_room_id']);
            $table->dropForeign(['booked_by']);
            $table->dropColumn([
                'operation_number',
                'patient_id',
                'visit_id',
                'surgeon_id',
                'assistant_surgeon_id',
                'anesthesiologist_id',
                'operation_room_id',
                'surgery_type',
                'procedure_name',
                'diagnosis',
                'indication',
                'scheduled_date',
                'scheduled_time',
                'estimated_duration',
                'status',
                'started_at',
                'completed_at',
                'pre_operative_notes',
                'operative_notes',
                'post_operative_notes',
                'complications',
                'notes',
                'booked_by',
            ]);
        });
    }
};
