<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Check database driver
        if (DB::getDriverName() === 'pgsql') {
            // PostgreSQL: Find the enum type name for visits.status column
            $result = DB::selectOne("
                SELECT t.typname 
                FROM pg_type t 
                JOIN pg_attribute a ON a.atttypid = t.oid 
                JOIN pg_class c ON a.attrelid = c.oid 
                WHERE c.relname = 'visits' 
                AND a.attname = 'status' 
                AND t.typtype = 'e'
            ");
            
            if ($result) {
                $enumType = $result->typname;
                // Add new value to enum type
                DB::statement("ALTER TYPE {$enumType} ADD VALUE IF NOT EXISTS 'admitted'");
            } else {
                // Fallback: If we can't find the enum type, use a different approach
                // This shouldn't happen if the table was created with enum, but handle it gracefully
                \Log::warning('Could not find enum type for visits.status column');
            }
        } else {
            // MySQL: Modify the enum to include 'admitted' status
            DB::statement("ALTER TABLE visits MODIFY COLUMN status ENUM(
                'checked_in',
                'waiting_nurse',
                'waiting_doctor',
                'in_consultation',
                'lab_requested',
                'prescription_issued',
                'pharmacy_dispensed',
                'billing_pending',
                'admitted',
                'checked_out'
            ) DEFAULT 'checked_in'");
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Note: PostgreSQL doesn't support removing enum values easily
        // This migration cannot be fully reversed for PostgreSQL
        if (DB::getDriverName() === 'mysql') {
            // MySQL: Revert to original enum (without 'admitted')
            DB::statement("ALTER TABLE visits MODIFY COLUMN status ENUM(
                'checked_in',
                'waiting_nurse',
                'waiting_doctor',
                'in_consultation',
                'lab_requested',
                'prescription_issued',
                'pharmacy_dispensed',
                'billing_pending',
                'checked_out'
            ) DEFAULT 'checked_in'");
        }
    }
};
