<?php

namespace App\Services;

use App\Models\SecurityEvent;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;

class SecurityEventService
{
    /**
     * Log a security event
     */
    public static function log(
        string $eventType,
        string $severity = SecurityEvent::SEVERITY_INFO,
        ?string $description = null,
        ?array $metadata = null,
        $resource = null
    ): SecurityEvent {
        return SecurityEvent::create([
            'user_id' => Auth::id(),
            'event_type' => $eventType,
            'severity' => $severity,
            'ip_address' => Request::ip(),
            'user_agent' => Request::userAgent(),
            'description' => $description,
            'metadata' => $metadata,
            'resource_type' => $resource ? get_class($resource) : null,
            'resource_id' => $resource ? $resource->id : null,
        ]);
    }

    /**
     * Log failed login attempt
     */
    public static function logFailedLogin(string $email): SecurityEvent
    {
        return self::log(
            SecurityEvent::TYPE_FAILED_LOGIN,
            SecurityEvent::SEVERITY_WARNING,
            "Failed login attempt for email: {$email}",
            ['email' => $email]
        );
    }

    /**
     * Log successful login
     */
    public static function logLogin(): SecurityEvent
    {
        return self::log(
            SecurityEvent::TYPE_LOGIN,
            SecurityEvent::SEVERITY_INFO,
            'User logged in successfully'
        );
    }

    /**
     * Log logout
     */
    public static function logLogout(): SecurityEvent
    {
        return self::log(
            SecurityEvent::TYPE_LOGOUT,
            SecurityEvent::SEVERITY_INFO,
            'User logged out'
        );
    }

    /**
     * Log unauthorized access attempt
     */
    public static function logUnauthorizedAccess(string $resource, ?string $action = null): SecurityEvent
    {
        return self::log(
            SecurityEvent::TYPE_UNAUTHORIZED_ACCESS,
            SecurityEvent::SEVERITY_CRITICAL,
            "Unauthorized access attempt to {$resource}" . ($action ? " ({$action})" : ''),
            ['resource' => $resource, 'action' => $action]
        );
    }

    /**
     * Log data access
     */
    public static function logDataAccess($resource, ?string $description = null): SecurityEvent
    {
        return self::log(
            SecurityEvent::TYPE_DATA_ACCESS,
            SecurityEvent::SEVERITY_INFO,
            $description ?? "Accessed " . get_class($resource),
            null,
            $resource
        );
    }

    /**
     * Log data modification
     */
    public static function logDataModification($resource, ?string $description = null): SecurityEvent
    {
        return self::log(
            SecurityEvent::TYPE_DATA_MODIFICATION,
            SecurityEvent::SEVERITY_WARNING,
            $description ?? "Modified " . get_class($resource),
            null,
            $resource
        );
    }
}


