<?php

namespace App\Services;

use Illuminate\Support\Facades\Auth;

class NavigationService
{
    /**
     * Get navigation menu items grouped by categories
     * 
     * @return array
     */
    public static function getMenuItems(): array
    {
        $user = Auth::user();
        
        if (!$user) {
            return [];
        }

        $menuItems = [];

        // Dashboard - always visible
        $menuItems[] = [
            'type' => 'link',
            'label' => 'Dashboard',
            'route' => 'hospital.dashboard',
            'routePattern' => 'hospital.dashboard',
            'icon' => null,
        ];

        // Patient Care Category
        $patientCareItems = [];
        if ($user->canViewPatients()) {
            $patientCareItems[] = [
                'label' => 'Patients',
                'route' => 'patients.index',
                'routePattern' => 'patients.*',
            ];
        }
        if ($user->isFrontDesk() || $user->isNurse() || $user->isDeveloper() || $user->isCmd()) {
            $patientCareItems[] = [
                'label' => 'Visits',
                'route' => 'visits.index',
                'routePattern' => 'visits.*',
            ];
        }
        if ($user->isDoctor() || $user->isNurse() || $user->isDeveloper() || $user->isCmd()) {
            $patientCareItems[] = [
                'label' => 'Admissions',
                'route' => 'admissions.index',
                'routePattern' => 'admissions.*',
            ];
        }
        if (!empty($patientCareItems)) {
            $menuItems[] = [
                'type' => 'dropdown',
                'label' => 'Patient Care',
                'items' => $patientCareItems,
            ];
        }

        // Clinical Services Category
        $clinicalItems = [];
        if ($user->isDoctor() || $user->isDeveloper() || $user->isCmd()) {
            $clinicalItems[] = [
                'label' => 'Consultations',
                'route' => 'doctor.consultations',
                'routePattern' => 'doctor.*',
            ];
        }
        if ($user->isNurse() || $user->isDeveloper() || $user->isCmd()) {
            $clinicalItems[] = [
                'label' => 'Nursing Station',
                'route' => 'nursing.vitals',
                'routePattern' => 'nursing.*',
            ];
            $clinicalItems[] = [
                'label' => 'Medication Log',
                'route' => 'medications.log',
                'routePattern' => 'medications.*',
            ];
        }
        if ($user->isDoctor() || $user->isLab() || $user->isDeveloper() || $user->isCmd()) {
            $clinicalItems[] = [
                'label' => 'Lab',
                'route' => 'lab.workflow',
                'routePattern' => 'lab.*',
            ];
        }
        if ($user->isDoctor() || $user->isNurse() || $user->isDeveloper() || $user->isCmd()) {
            $clinicalItems[] = [
                'label' => 'Surgeries',
                'route' => 'surgeries.index',
                'routePattern' => 'surgeries.*',
            ];
        }
        if (!empty($clinicalItems)) {
            $menuItems[] = [
                'type' => 'dropdown',
                'label' => 'Clinical Services',
                'items' => $clinicalItems,
            ];
        }

        // Pharmacy & Billing Category
        $pharmacyBillingItems = [];
        if ($user->canViewPrescriptions()) {
            $pharmacyBillingItems[] = [
                'label' => 'Prescriptions',
                'route' => 'prescriptions.index',
                'routePattern' => 'prescriptions.*',
            ];
        }
        if (($user->isPharmacy() || $user->isDeveloper() || $user->isCmd())) {
            $pharmacyBillingItems[] = [
                'label' => 'Drug Management',
                'route' => 'drugs.index',
                'routePattern' => 'drugs.*',
            ];
            $pharmacyBillingItems[] = [
                'label' => 'Inventory',
                'route' => 'inventory.index',
                'routePattern' => 'inventory.*',
            ];
        }
        if ($user->canViewBills()) {
            $pharmacyBillingItems[] = [
                'label' => 'Billing',
                'route' => $user->canManageBills() ? 'billing.visits' : 'bills.index',
                'routePattern' => 'billing.*|bills.*',
            ];
        }
        if (!empty($pharmacyBillingItems)) {
            $menuItems[] = [
                'type' => 'dropdown',
                'label' => 'Pharmacy & Billing',
                'items' => $pharmacyBillingItems,
            ];
        }

        // Appointments - single link (could be in Patient Care or separate)
        if ($user->canViewAppointments()) {
            $menuItems[] = [
                'type' => 'link',
                'label' => 'Appointments',
                'route' => 'appointments.index',
                'routePattern' => 'appointments.*',
            ];
        }

        // Administration Category
        $adminItems = [];
        if ($user->isDeveloper() || $user->isCmd()) {
            $adminItems[] = [
                'label' => 'Services',
                'route' => 'services.index',
                'routePattern' => 'services.*',
            ];
        }
        if ($user->canAccessAI()) {
            $adminItems[] = [
                'label' => 'AI Features',
                'route' => 'ai.drug-interactions',
                'routePattern' => 'ai.*',
            ];
        }
        if (!empty($adminItems)) {
            $menuItems[] = [
                'type' => 'dropdown',
                'label' => 'Administration',
                'items' => $adminItems,
            ];
        }

        return $menuItems;
    }

    /**
     * Check if a route pattern matches the current route
     * 
     * @param string $pattern
     * @return bool
     */
    public static function routeMatches(string $pattern): bool
    {
        return request()->routeIs($pattern);
    }

    /**
     * Check if any item in a dropdown is active
     * 
     * @param array $items
     * @return bool
     */
    public static function isDropdownActive(array $items): bool
    {
        foreach ($items as $item) {
            if (request()->routeIs($item['routePattern'])) {
                return true;
            }
        }
        return false;
    }
}

