<?php

namespace App\Services;

use App\Models\Patient;
use App\Models\Appointment;
use App\Models\MedicalRecord;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;

class AIPredictiveAnalyticsService
{
    private $apiKey;
    private $baseUrl = 'https://api.openai.com/v1/chat/completions';

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key');
    }

    /**
     * Predict patient readmission risk
     */
    public function predictReadmissionRisk(int $patientId): array
    {
        $cacheKey = 'readmission_risk_' . $patientId;
        
        return Cache::remember($cacheKey, 3600, function () use ($patientId) {
            return $this->analyzeReadmissionRisk($patientId);
        });
    }

    /**
     * Analyze readmission risk using AI
     */
    private function analyzeReadmissionRisk(int $patientId): array
    {
        $patient = Patient::with(['medicalRecords', 'appointments', 'prescriptions'])->find($patientId);
        
        if (!$patient) {
            return ['risk_level' => 'unknown', 'factors' => [], 'recommendations' => []];
        }

        $patientData = $this->preparePatientData($patient);
        $prompt = $this->buildReadmissionPrompt($patientData);

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post($this->baseUrl, [
                'model' => 'gpt-4',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a medical AI assistant specializing in predictive analytics for healthcare. Analyze patient data to predict readmission risk and provide evidence-based recommendations.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 1000,
                'temperature' => 0.2,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return $this->parseReadmissionResponse($result['choices'][0]['message']['content']);
            }
        } catch (\Exception $e) {
            \Log::error('AI Readmission Prediction Failed: ' . $e->getMessage());
        }

        return $this->getFallbackReadmissionAnalysis($patient);
    }

    /**
     * Prepare patient data for analysis
     */
    private function preparePatientData(Patient $patient): array
    {
        $recentRecords = $patient->medicalRecords()
            ->orderBy('visit_date', 'desc')
            ->limit(5)
            ->get();

        $recentAppointments = $patient->appointments()
            ->orderBy('appointment_date', 'desc')
            ->limit(10)
            ->get();

        return [
            'patient_info' => [
                'age' => $patient->age,
                'gender' => $patient->gender,
                'medical_history' => $patient->medical_history,
                'allergies' => $patient->allergies,
                'blood_type' => $patient->blood_type,
            ],
            'recent_visits' => $recentRecords->map(function ($record) {
                return [
                    'date' => $record->visit_date->format('Y-m-d'),
                    'type' => $record->visit_type,
                    'diagnosis' => $record->diagnosis,
                    'symptoms' => $record->symptoms,
                ];
            })->toArray(),
            'appointment_history' => $recentAppointments->map(function ($appointment) {
                return [
                    'date' => $appointment->appointment_date->format('Y-m-d'),
                    'type' => $appointment->type,
                    'status' => $appointment->status,
                ];
            })->toArray(),
            'total_visits' => $patient->medicalRecords()->count(),
            'total_appointments' => $patient->appointments()->count(),
        ];
    }

    /**
     * Build readmission prediction prompt
     */
    private function buildReadmissionPrompt(array $patientData): string
    {
        $prompt = "Analyze the following patient data for readmission risk:\n\n";
        
        $prompt .= "PATIENT DEMOGRAPHICS:\n";
        $prompt .= "- Age: " . $patientData['patient_info']['age'] . "\n";
        $prompt .= "- Gender: " . $patientData['patient_info']['gender'] . "\n";
        $prompt .= "- Blood Type: " . ($patientData['patient_info']['blood_type'] ?? 'Unknown') . "\n";

        $prompt .= "\nMEDICAL HISTORY:\n";
        $prompt .= "- Medical History: " . ($patientData['patient_info']['medical_history'] ?? 'None recorded') . "\n";
        $prompt .= "- Allergies: " . ($patientData['patient_info']['allergies'] ?? 'None recorded') . "\n";

        $prompt .= "\nRECENT VISITS (" . count($patientData['recent_visits']) . "):\n";
        foreach ($patientData['recent_visits'] as $visit) {
            $prompt .= "- " . $visit['date'] . ": " . $visit['type'] . " - " . ($visit['diagnosis'] ?? 'No diagnosis') . "\n";
        }

        $prompt .= "\nVISIT FREQUENCY:\n";
        $prompt .= "- Total visits: " . $patientData['total_visits'] . "\n";
        $prompt .= "- Total appointments: " . $patientData['total_appointments'] . "\n";

        $prompt .= "\nPlease provide:\n";
        $prompt .= "1. Risk level (low, moderate, high)\n";
        $prompt .= "2. Key risk factors identified\n";
        $prompt .= "3. Confidence in prediction (percentage)\n";
        $prompt .= "4. Preventive recommendations\n";
        $prompt .= "5. Follow-up suggestions\n\n";
        $prompt .= "Format as JSON with keys: risk_level, risk_factors, confidence, recommendations, follow_up_suggestions";

        return $prompt;
    }

    /**
     * Parse readmission response
     */
    private function parseReadmissionResponse(string $response): array
    {
        try {
            if (preg_match('/\{.*\}/s', $response, $matches)) {
                $jsonData = json_decode($matches[0], true);
                if ($jsonData) {
                    return array_merge($jsonData, ['ai_generated' => true]);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to parse AI readmission response: ' . $e->getMessage());
        }

        return [
            'risk_level' => 'moderate',
            'risk_factors' => ['Limited data for analysis'],
            'confidence' => 50,
            'recommendations' => ['Regular follow-up appointments', 'Monitor symptoms closely'],
            'follow_up_suggestions' => ['Schedule follow-up in 1-2 weeks'],
            'ai_generated' => false,
        ];
    }

    /**
     * Get fallback readmission analysis
     */
    private function getFallbackReadmissionAnalysis(Patient $patient): array
    {
        $visitCount = $patient->medicalRecords()->count();
        $riskLevel = 'low';
        
        if ($visitCount > 5) {
            $riskLevel = 'high';
        } elseif ($visitCount > 2) {
            $riskLevel = 'moderate';
        }

        return [
            'risk_level' => $riskLevel,
            'risk_factors' => ['Visit frequency', 'Medical history complexity'],
            'confidence' => 60,
            'recommendations' => ['Regular monitoring', 'Preventive care'],
            'follow_up_suggestions' => ['Schedule regular checkups'],
            'ai_generated' => false,
        ];
    }

    /**
     * Predict appointment no-show probability
     */
    public function predictNoShowProbability(int $appointmentId): array
    {
        $cacheKey = 'no_show_probability_' . $appointmentId;
        
        return Cache::remember($cacheKey, 1800, function () use ($appointmentId) {
            return $this->analyzeNoShowRisk($appointmentId);
        });
    }

    /**
     * Analyze no-show risk
     */
    private function analyzeNoShowRisk(int $appointmentId): array
    {
        $appointment = Appointment::with(['patient', 'doctor'])->find($appointmentId);
        
        if (!$appointment) {
            return ['probability' => 0, 'factors' => [], 'recommendations' => []];
        }

        $patient = $appointment->patient;
        $noShowHistory = $patient->appointments()
            ->where('status', 'no_show')
            ->count();
        
        $totalAppointments = $patient->appointments()->count();
        $noShowRate = $totalAppointments > 0 ? ($noShowHistory / $totalAppointments) * 100 : 0;

        $prompt = "Analyze no-show probability for this appointment:\n\n";
        $prompt .= "PATIENT INFO:\n";
        $prompt .= "- Age: " . $patient->age . "\n";
        $prompt .= "- Gender: " . $patient->gender . "\n";
        $prompt .= "- Phone: " . ($patient->phone ?? 'Not provided') . "\n";

        $prompt .= "\nAPPOINTMENT DETAILS:\n";
        $prompt .= "- Date: " . $appointment->appointment_date->format('Y-m-d') . "\n";
        $prompt .= "- Time: " . $appointment->appointment_time->format('H:i') . "\n";
        $prompt .= "- Type: " . $appointment->type . "\n";
        $prompt .= "- Doctor: " . $appointment->doctor->name . "\n";

        $prompt .= "\nPATIENT HISTORY:\n";
        $prompt .= "- Total appointments: " . $totalAppointments . "\n";
        $prompt .= "- No-show rate: " . round($noShowRate, 2) . "%\n";
        $prompt .= "- Previous no-shows: " . $noShowHistory . "\n";

        $prompt .= "\nProvide:\n";
        $prompt .= "1. No-show probability (0-100%)\n";
        $prompt .= "2. Risk factors\n";
        $prompt .= "3. Intervention recommendations\n";
        $prompt .= "Format as JSON with keys: probability, risk_factors, recommendations";

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(20)->post($this->baseUrl, [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a healthcare analytics AI. Analyze appointment no-show risk based on patient history and appointment details.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 500,
                'temperature' => 0.3,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return $this->parseNoShowResponse($result['choices'][0]['message']['content']);
            }
        } catch (\Exception $e) {
            \Log::error('AI No-Show Prediction Failed: ' . $e->getMessage());
        }

        return [
            'probability' => min($noShowRate + 10, 100),
            'risk_factors' => ['Historical no-show rate', 'Limited contact information'],
            'recommendations' => ['Send reminder messages', 'Confirm appointment day before'],
            'ai_generated' => false,
        ];
    }

    /**
     * Parse no-show response
     */
    private function parseNoShowResponse(string $response): array
    {
        try {
            if (preg_match('/\{.*\}/s', $response, $matches)) {
                $jsonData = json_decode($matches[0], true);
                if ($jsonData) {
                    return array_merge($jsonData, ['ai_generated' => true]);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to parse AI no-show response: ' . $e->getMessage());
        }

        return [
            'probability' => 25,
            'risk_factors' => ['Standard risk assessment'],
            'recommendations' => ['Send appointment reminders'],
            'ai_generated' => false,
        ];
    }

    /**
     * Generate hospital analytics insights
     */
    public function generateAnalyticsInsights(): array
    {
        $cacheKey = 'hospital_analytics_insights';
        
        return Cache::remember($cacheKey, 7200, function () {
            return $this->generateAIInsights();
        });
    }

    /**
     * Generate AI-powered insights
     */
    private function generateAIInsights(): array
    {
        $data = [
            'total_patients' => Patient::count(),
            'total_appointments' => Appointment::count(),
            'pending_bills' => \App\Models\Bill::where('status', 'pending')->count(),
            'active_prescriptions' => \App\Models\Prescription::where('status', 'active')->count(),
        ];

        $prompt = "Analyze the following hospital metrics and provide insights:\n\n";
        $prompt .= "METRICS:\n";
        $prompt .= "- Total patients: " . $data['total_patients'] . "\n";
        $prompt .= "- Total appointments: " . $data['total_appointments'] . "\n";
        $prompt .= "- Pending bills: " . $data['pending_bills'] . "\n";
        $prompt .= "- Active prescriptions: " . $data['active_prescriptions'] . "\n\n";
        $prompt .= "Provide:\n";
        $prompt .= "1. Key insights\n";
        $prompt .= "2. Areas for improvement\n";
        $prompt .= "3. Recommendations\n";
        $prompt .= "Format as JSON with keys: insights, improvements, recommendations";

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(20)->post($this->baseUrl, [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a healthcare analytics AI providing insights on hospital operations and suggesting improvements.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 800,
                'temperature' => 0.4,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return $this->parseInsightsResponse($result['choices'][0]['message']['content']);
            }
        } catch (\Exception $e) {
            \Log::error('AI Analytics Insights Failed: ' . $e->getMessage());
        }

        return [
            'insights' => ['Hospital is operating normally'],
            'improvements' => ['Monitor patient satisfaction', 'Optimize appointment scheduling'],
            'recommendations' => ['Regular performance reviews', 'Staff training programs'],
            'ai_generated' => false,
        ];
    }

    /**
     * Parse insights response
     */
    private function parseInsightsResponse(string $response): array
    {
        try {
            if (preg_match('/\{.*\}/s', $response, $matches)) {
                $jsonData = json_decode($matches[0], true);
                if ($jsonData) {
                    return array_merge($jsonData, ['ai_generated' => true]);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to parse AI insights response: ' . $e->getMessage());
        }

        return [
            'insights' => ['Hospital is operating normally'],
            'improvements' => ['Monitor patient satisfaction', 'Optimize appointment scheduling'],
            'recommendations' => ['Regular performance reviews', 'Staff training programs'],
            'ai_generated' => false,
        ];
    }
}