<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    public function patient()
    {
        return $this->hasOne(Patient::class);
    }

    public function appointmentsAsDoctor()
    {
        return $this->hasMany(Appointment::class, 'doctor_id');
    }

    public function medicalRecordsAsDoctor()
    {
        return $this->hasMany(MedicalRecord::class, 'doctor_id');
    }

    public function medicationsAsPrescriber()
    {
        return $this->hasMany(Medication::class, 'prescribed_by');
    }

    public function prescriptions()
    {
        return $this->hasMany(Prescription::class, 'doctor_id');
    }

    public function consultations()
    {
        return $this->hasMany(Consultation::class, 'doctor_id');
    }

    public function billsCreated()
    {
        return $this->hasMany(Bill::class, 'created_by');
    }

    public function paymentsProcessed()
    {
        return $this->hasMany(Payment::class, 'processed_by');
    }

    /**
     * Check if user has a specific role
     */
    public function hasRole(string $role): bool
    {
        return $this->role === $role;
    }

    /**
     * Check if user is a doctor
     */
    public function isDoctor(): bool
    {
        return $this->role === 'doctor';
    }

    /**
     * Check if user is a nurse
     */
    public function isNurse(): bool
    {
        return $this->role === 'nurse';
    }

    /**
     * Check if user is front desk staff
     */
    public function isFrontDesk(): bool
    {
        return $this->role === 'front_desk';
    }

    /**
     * Check if user is a developer
     */
    public function isDeveloper(): bool
    {
        return $this->role === 'developer';
    }

    /**
     * Check if user is CMD (Chief Medical Director)
     */
    public function isCmd(): bool
    {
        return $this->role === 'cmd';
    }

    /**
     * Check if user is pharmacy staff
     */
    public function isPharmacy(): bool
    {
        return $this->role === 'pharmacy';
    }

    /**
     * Check if user is billing staff
     */
    public function isBilling(): bool
    {
        return $this->role === 'billing';
    }

    /**
     * Check if user is lab staff
     */
    public function isLab(): bool
    {
        return $this->role === 'lab';
    }

    /**
     * Check if user is staff
     */
    public function isStaff(): bool
    {
        return $this->role === 'staff';
    }

    /**
     * Check if user can view patients
     * Based on PatientPolicy, all authenticated users can view patients
     */
    public function canViewPatients(): bool
    {
        return true;
    }

    /**
     * Check if user can create users
     * Only developers can create users
     */
    public function canCreateUsers(): bool
    {
        return $this->isDeveloper();
    }

    /**
     * Check if user can manage patients (create/update)
     * Front desk, developer, and cmd can manage patients (doctors cannot create patients)
     */
    public function canManagePatients(): bool
    {
        return $this->isFrontDesk() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can view appointments
     */
    public function canViewAppointments(): bool
    {
        return $this->isDoctor() || $this->isFrontDesk() || $this->isNurse() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can manage appointments (create/update/delete)
     * Doctors, front desk, developer, and cmd can manage appointments
     */
    public function canManageAppointments(): bool
    {
        return $this->isDoctor() || $this->isFrontDesk() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can view prescriptions
     */
    public function canViewPrescriptions(): bool
    {
        return $this->isDoctor() || $this->isNurse() || $this->isPharmacy() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can manage prescriptions (create/update)
     * Doctors, developer, and cmd can manage prescriptions
     */
    public function canManagePrescriptions(): bool
    {
        return $this->isDoctor() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can view bills
     * Doctors should NOT have access to billing
     */
    public function canViewBills(): bool
    {
        if ($this->isDoctor()) {
            return false;
        }
        return $this->isFrontDesk() || $this->canManageBills() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can manage bills
     */
    public function canManageBills(): bool
    {
        return $this->isFrontDesk() || $this->isBilling() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can view consultations
     */
    public function canViewConsultations(): bool
    {
        return $this->isDoctor() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can access AI features
     */
    public function canAccessAI(): bool
    {
        return $this->isDoctor() || $this->isDeveloper() || $this->isCmd();
    }

    /**
     * Check if user can enter chief complaint
     * Doctors, developers, and cmd can enter chief complaint
     */
    public function canEnterChiefComplaint(): bool
    {
        return $this->isDoctor() || $this->isDeveloper() || $this->isCmd();
    }

}
