<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Service extends Model
{
    use HasFactory;

    protected $fillable = [
        'code',
        'name',
        'category',
        'subcategory',
        'service_key',
        'price',
        'unit',
        'description',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get service by category and service key
     */
    public static function getByKey(string $category, ?string $serviceKey = null): ?self
    {
        $query = self::where('category', $category)
            ->where('is_active', true);
        
        if ($serviceKey) {
            $query->where(function($q) use ($serviceKey) {
                $q->where('service_key', $serviceKey)
                  ->orWhereNull('service_key'); // Fallback to general service
            });
        } else {
            $query->whereNull('service_key'); // General service for category
        }
        
        return $query->orderBy('sort_order')->first();
    }

    /**
     * Get consultation service by type
     */
    public static function getConsultationService(?string $consultationType = null): ?self
    {
        return self::getByKey('consultation', $consultationType);
    }

    /**
     * Get lab test service by test name
     */
    public static function getLabTestService(?string $testName = null): ?self
    {
        // First try to find exact match by test name
        if ($testName) {
            $service = self::where('category', 'lab_test')
                ->where('is_active', true)
                ->where('service_key', $testName)
                ->first();
            
            if ($service) {
                return $service;
            }
        }
        
        // Fallback to general lab test service
        return self::getByKey('lab_test');
    }

    /**
     * Get prescription service
     */
    public static function getPrescriptionService(): ?self
    {
        return self::getByKey('prescription');
    }

    /**
     * Get formatted price
     */
    public function getFormattedPriceAttribute(): string
    {
        return config('app.currency_symbol') . number_format($this->price, 2);
    }

    /**
     * Scope for active services
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for category
     */
    public function scopeCategory($query, string $category)
    {
        return $query->where('category', $category);
    }
}
