<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\Medication;

class Prescription extends Model
{
    use HasFactory;

    protected $fillable = [
        'patient_id',
        'visit_id',
        'doctor_id',
        'drug_id',
        'prescription_number',
        'dosage',
        'frequency',
        'duration',
        'quantity',
        'instructions',
        'prescription_date',
        'valid_until',
        'status',
        'notes',
        'refill_count',
        'max_refills',
        'pharmacy_notes',
    ];

    protected $casts = [
        'prescription_date' => 'date',
        'valid_until' => 'date',
    ];

    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function doctor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'doctor_id');
    }

    public function drug(): BelongsTo
    {
        return $this->belongsTo(Drug::class);
    }

    public function visit(): BelongsTo
    {
        return $this->belongsTo(Visit::class);
    }

    public function dispensings(): HasMany
    {
        return $this->hasMany(PrescriptionDispensing::class);
    }

    /**
     * Boot the model and sync medications
     */
    protected static function boot()
    {
        parent::boot();

        static::created(function ($prescription) {
            $prescription->syncMedication();
        });

        static::updated(function ($prescription) {
            $prescription->syncMedication();
        });
    }

    /**
     * Sync medication record from prescription
     */
    public function syncMedication(): void
    {
        // Load drug relationship if not already loaded
        if (!$this->relationLoaded('drug')) {
            $this->load('drug');
        }

        if (!$this->drug) {
            return; // Can't sync without drug information
        }

        // Only sync if prescription is active
        if ($this->status !== 'active') {
            // If prescription is not active (expired, cancelled, dispensed), mark corresponding medication as completed
            Medication::where('patient_id', $this->patient_id)
                ->where('name', $this->drug->name)
                ->where('status', 'active')
                ->update(['status' => 'completed']);
            return;
        }

        // Check if prescription has expired (double-check even if status is active)
        if ($this->valid_until < now()->startOfDay()) {
            // Auto-update status to expired
            $this->status = 'expired';
            $this->saveQuietly(); // Save without triggering events to avoid recursion
            // Mark medication as completed
            Medication::where('patient_id', $this->patient_id)
                ->where('name', $this->drug->name)
                ->where('status', 'active')
                ->update(['status' => 'completed']);
            return;
        }

        // Calculate end date based on duration
        $startDate = $this->prescription_date;
        $endDate = $this->valid_until;

        // Find or create medication record
        $medication = Medication::updateOrCreate(
            [
                'patient_id' => $this->patient_id,
                'name' => $this->drug->name,
                'status' => 'active',
            ],
            [
                'prescribed_by' => $this->doctor_id,
                'dosage' => $this->dosage,
                'frequency' => $this->frequency,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'instructions' => $this->instructions,
                'notes' => $this->notes,
            ]
        );
    }

    public function getPrescriptionNumberAttribute(): string
    {
        return 'RX' . str_pad($this->id, 6, '0', STR_PAD_LEFT);
    }

    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->status) {
            'active' => 'green',
            'dispensed' => 'blue',
            'expired' => 'red',
            'cancelled' => 'gray',
            default => 'yellow'
        };
    }

    public function canBeRefilled(): bool
    {
        return $this->status === 'active' && 
               $this->refill_count < $this->max_refills && 
               $this->valid_until > now();
    }

    /**
     * Check if prescription is expired
     */
    public function isExpired(): bool
    {
        return $this->status === 'active' && 
               $this->valid_until < now()->startOfDay();
    }

    /**
     * Scope to get expired prescriptions
     */
    public function scopeExpired($query)
    {
        return $query->where('status', 'active')
                     ->where('valid_until', '<', now()->startOfDay());
    }

    /**
     * Scope to get active prescriptions
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active')
                     ->where('valid_until', '>=', now()->startOfDay());
    }
}