<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Patient extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id', // nullable - tracks which staff member registered the patient (not patient's own account)
        'patient_id',
        'first_name',
        'last_name',
        'date_of_birth',
        'gender',
        'email',
        'phone',
        'address',
        'emergency_contact_name',
        'emergency_contact_phone',
        'insurance_provider',
        'insurance_number',
        'medical_history',
        'allergies',
        'blood_type',
        'height',
        'weight',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'height' => 'decimal:2',
        'weight' => 'decimal:2',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function medicalRecords(): HasMany
    {
        return $this->hasMany(MedicalRecord::class);
    }

    public function appointments(): HasMany
    {
        return $this->hasMany(Appointment::class);
    }

    public function medications(): HasMany
    {
        return $this->hasMany(Medication::class);
    }

    public function prescriptions(): HasMany
    {
        return $this->hasMany(Prescription::class);
    }

    public function consultations(): HasMany
    {
        return $this->hasMany(Consultation::class);
    }

    public function bills(): HasMany
    {
        return $this->hasMany(Bill::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    public function visits(): HasMany
    {
        return $this->hasMany(Visit::class);
    }

    public function auditLogs(): HasMany
    {
        return $this->hasMany(AuditLog::class);
    }

    public function admissions(): HasMany
    {
        return $this->hasMany(Admission::class);
    }

    /**
     * Get active admission
     */
    public function getActiveAdmissionAttribute(): ?Admission
    {
        return $this->admissions()
            ->where('status', 'admitted')
            ->latest('admission_date')
            ->first();
    }

    public function getFullNameAttribute(): string
    {
        $firstName = ucwords(strtolower($this->first_name ?? ''));
        $lastName = ucwords(strtolower($this->last_name ?? ''));
        return trim($firstName . ' ' . $lastName);
    }

    /**
     * Get capitalized address for display
     */
    public function getDisplayAddressAttribute(): ?string
    {
        return $this->address ? ucwords(strtolower($this->address)) : null;
    }

    /**
     * Get capitalized emergency contact name for display
     */
    public function getDisplayEmergencyContactNameAttribute(): ?string
    {
        return $this->emergency_contact_name ? ucwords(strtolower($this->emergency_contact_name)) : null;
    }

    /**
     * Get capitalized insurance provider for display
     */
    public function getDisplayInsuranceProviderAttribute(): ?string
    {
        return $this->insurance_provider ? ucwords(strtolower($this->insurance_provider)) : null;
    }

    /**
     * Get active visit (not checked out)
     */
    public function getActiveVisitAttribute(): ?Visit
    {
        return $this->visits()
            ->where('status', '!=', Visit::STATUS_CHECKED_OUT)
            ->whereNull('checked_out_at')
            ->latest('checked_in_at')
            ->first();
    }

    public function getAgeAttribute(): int
    {
        return $this->date_of_birth->age;
    }
}

