<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class OperationRoom extends Model
{
    use HasFactory;

    protected $fillable = [
        'room_number',
        'name',
        'description',
        'status',
        'equipment',
        'capacity',
        'is_active',
    ];

    protected $casts = [
        'equipment' => 'array',
        'capacity' => 'integer',
        'is_active' => 'boolean',
    ];

    // Status constants
    const STATUS_AVAILABLE = 'available';
    const STATUS_OCCUPIED = 'occupied';
    const STATUS_MAINTENANCE = 'maintenance';
    const STATUS_UNAVAILABLE = 'unavailable';

    public function operations(): HasMany
    {
        return $this->hasMany(Operation::class);
    }

    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            self::STATUS_AVAILABLE => 'green',
            self::STATUS_OCCUPIED => 'red',
            self::STATUS_MAINTENANCE => 'yellow',
            self::STATUS_UNAVAILABLE => 'gray',
            default => 'gray'
        };
    }

    public function getStatusLabelAttribute(): string
    {
        return ucwords(str_replace('_', ' ', $this->status));
    }

    public function getDisplayNameAttribute(): string
    {
        return ucwords(strtolower($this->name));
    }

    /**
     * Check if room is available for a given date and time
     */
    public function isAvailableFor($date, $time, $duration, $excludeOperationId = null): bool
    {
        if ($this->status !== self::STATUS_AVAILABLE || !$this->is_active) {
            return false;
        }

        $startDateTime = \Carbon\Carbon::parse($date . ' ' . $time);
        $endDateTime = $startDateTime->copy()->addMinutes($duration);

        $conflictingOperations = Operation::where('operation_room_id', $this->id)
            ->where('status', '!=', 'cancelled')
            ->where('status', '!=', 'completed')
            ->where(function ($query) use ($startDateTime, $endDateTime) {
                $query->whereBetween('scheduled_date', [
                    $startDateTime->toDateString(),
                    $endDateTime->toDateString()
                ])
                ->orWhere(function ($q) use ($startDateTime, $endDateTime) {
                    $q->whereDate('scheduled_date', $startDateTime->toDateString())
                      ->where(function ($subQ) use ($startDateTime, $endDateTime) {
                          $subQ->whereTime('scheduled_time', '>=', $startDateTime->format('H:i:s'))
                               ->whereTime('scheduled_time', '<', $endDateTime->format('H:i:s'));
                      });
                });
            });

        if ($excludeOperationId) {
            $conflictingOperations->where('id', '!=', $excludeOperationId);
        }

        return $conflictingOperations->count() === 0;
    }
}
