<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class Operation extends Model
{
    use HasFactory;

    protected $fillable = [
        'operation_number',
        'patient_id',
        'visit_id',
        'surgeon_id',
        'assistant_surgeon_id',
        'anesthesiologist_id',
        'operation_room_id',
        'surgery_type',
        'procedure_name',
        'diagnosis',
        'indication',
        'scheduled_date',
        'scheduled_time',
        'estimated_duration',
        'status',
        'started_at',
        'completed_at',
        'pre_operative_notes',
        'operative_notes',
        'post_operative_notes',
        'complications',
        'notes',
        'booked_by',
    ];

    protected $casts = [
        'scheduled_date' => 'date',
        'scheduled_time' => 'string',
        'estimated_duration' => 'integer',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
    ];

    // Status constants
    const STATUS_SCHEDULED = 'scheduled';
    const STATUS_PREPARING = 'preparing';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';
    const STATUS_POSTPONED = 'postponed';

    /**
     * Boot the model and generate operation number
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($operation) {
            if (empty($operation->operation_number)) {
                $operation->operation_number = static::generateOperationNumber();
            }
        });
    }

    /**
     * Generate unique operation number
     */
    public static function generateOperationNumber(): string
    {
        $year = now()->format('Y');
        $lastOperation = static::whereYear('created_at', $year)
            ->orderBy('id', 'desc')
            ->first();
        
        $number = $lastOperation ? ((int) substr($lastOperation->operation_number, -6)) + 1 : 1;
        
        return 'OP-' . $year . '-' . str_pad($number, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Relationships
     */
    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function visit(): BelongsTo
    {
        return $this->belongsTo(Visit::class);
    }

    public function surgeon(): BelongsTo
    {
        return $this->belongsTo(User::class, 'surgeon_id');
    }

    public function assistantSurgeon(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assistant_surgeon_id');
    }

    public function anesthesiologist(): BelongsTo
    {
        return $this->belongsTo(User::class, 'anesthesiologist_id');
    }

    public function operationRoom(): BelongsTo
    {
        return $this->belongsTo(OperationRoom::class);
    }

    public function bookedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'booked_by');
    }

    /**
     * Accessors
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            self::STATUS_SCHEDULED => 'yellow',
            self::STATUS_PREPARING => 'blue',
            self::STATUS_IN_PROGRESS => 'purple',
            self::STATUS_COMPLETED => 'green',
            self::STATUS_CANCELLED => 'red',
            self::STATUS_POSTPONED => 'orange',
            default => 'gray'
        };
    }

    public function getStatusLabelAttribute(): string
    {
        return ucwords(str_replace('_', ' ', $this->status));
    }

    public function getProcedureNameAttribute(): string
    {
        return ucwords(strtolower($this->attributes['procedure_name'] ?? ''));
    }

    public function getSurgeryTypeAttribute(): string
    {
        return ucwords(strtolower($this->attributes['surgery_type'] ?? ''));
    }

    /**
     * Check if operation can be started
     */
    public function canStart(): bool
    {
        return in_array($this->status, [self::STATUS_SCHEDULED, self::STATUS_PREPARING]);
    }

    /**
     * Check if operation can be completed
     */
    public function canComplete(): bool
    {
        return $this->status === self::STATUS_IN_PROGRESS;
    }

    /**
     * Start the operation
     */
    public function start(): void
    {
        if ($this->canStart()) {
            $this->update([
                'status' => self::STATUS_IN_PROGRESS,
                'started_at' => now(),
            ]);

            // Update operation room status
            if ($this->operationRoom) {
                $this->operationRoom->update(['status' => OperationRoom::STATUS_OCCUPIED]);
            }
        }
    }

    /**
     * Complete the operation
     */
    public function complete(): void
    {
        if ($this->canComplete()) {
            $this->update([
                'status' => self::STATUS_COMPLETED,
                'completed_at' => now(),
            ]);

            // Update operation room status
            if ($this->operationRoom) {
                $this->operationRoom->update(['status' => OperationRoom::STATUS_AVAILABLE]);
            }
        }
    }
}
