<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Drug extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'generic_name',
        'manufacturer',
        'category',
        'category_id',
        'description',
        'dosage_form',
        'strength',
        'unit',
        'prescription_required',
        'side_effects',
        'contraindications',
        'interactions',
        'storage_conditions',
        'expiry_alert_days',
        'is_active',
    ];

    protected $casts = [
        'prescription_required' => 'boolean',
        'is_active' => 'boolean',
        'side_effects' => 'array',
        'contraindications' => 'array',
        'interactions' => 'array',
    ];

    public function prescriptions(): HasMany
    {
        return $this->hasMany(Prescription::class);
    }

    public function pharmacyStocks(): HasMany
    {
        return $this->hasMany(PharmacyStock::class);
    }

    /**
     * Get total quantity available across all stock entries
     */
    public function getTotalQuantityAttribute(): int
    {
        return $this->pharmacyStocks()->sum('quantity') ?? 0;
    }

    /**
     * Get total quantity excluding expired stock
     */
    public function getAvailableQuantityAttribute(): int
    {
        return $this->pharmacyStocks()
            ->where(function($query) {
                $query->whereNull('expiry_date')
                      ->orWhere('expiry_date', '>', now());
            })
            ->sum('quantity') ?? 0;
    }

    /**
     * Get the category that owns the drug
     */
    public function drugCategory(): BelongsTo
    {
        return $this->belongsTo(DrugCategory::class, 'category_id');
    }

    /**
     * Get the category name (from relationship or fallback to category column)
     */
    public function getCategoryNameAttribute(): string
    {
        return $this->drugCategory ? $this->drugCategory->name : ($this->category ?? '');
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to filter by category
     */
    public function scopeByCategory($query, $categoryId)
    {
        return $query->where('category_id', $categoryId);
    }

    public function getFullNameAttribute(): string
    {
        return "{$this->name} ({$this->generic_name})";
    }
}