<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class Bill extends Model
{
    use HasFactory;

    protected $fillable = [
        'bill_number',
        'patient_id',
        'visit_id',
        'billable_type',
        'billable_id',
        'subtotal',
        'tax_amount',
        'discount_amount',
        'total_amount',
        'due_date',
        'status',
        'notes',
        'created_by',
        'billing_period_start',
        'billing_period_end',
        'insurance_covered',
        'patient_responsibility',
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'patient_responsibility' => 'decimal:2',
        'due_date' => 'date',
        'billing_period_start' => 'date',
        'billing_period_end' => 'date',
        'insurance_covered' => 'boolean',
    ];

    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function visit(): BelongsTo
    {
        return $this->belongsTo(Visit::class);
    }

    public function billable(): MorphTo
    {
        return $this->morphTo();
    }

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getBillNumberAttribute(): string
    {
        return 'BILL' . str_pad($this->id, 6, '0', STR_PAD_LEFT);
    }

    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->status) {
            'paid' => 'green',
            'pending' => 'yellow',
            'overdue' => 'red',
            'cancelled' => 'gray',
            'partial' => 'blue',
            default => 'gray'
        };
    }

    public function getPaidAmountAttribute(): float
    {
        return $this->payments()->where('status', 'completed')->sum('amount');
    }

    public function getBalanceAttribute(): float
    {
        return $this->total_amount - $this->paid_amount;
    }

    public function isPaid(): bool
    {
        return $this->balance <= 0;
    }

    public function isOverdue(): bool
    {
        return $this->due_date < now() && !$this->isPaid();
    }
}