<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class AuditLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'patient_id',
        'visit_id',
        'model_type',
        'model_id',
        'action',
        'field_name',
        'old_value',
        'new_value',
        'description',
        'ip_address',
        'user_agent',
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Relationships
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function visit(): BelongsTo
    {
        return $this->belongsTo(Visit::class);
    }

    public function model(): MorphTo
    {
        return $this->morphTo('model');
    }

    /**
     * Get human-readable action label
     */
    public function getActionLabelAttribute(): string
    {
        return match($this->action) {
            'created' => 'Created',
            'updated' => 'Updated',
            'deleted' => 'Deleted',
            default => ucfirst($this->action),
        };
    }

    /**
     * Get human-readable field label
     */
    public function getFieldLabelAttribute(): string
    {
        return match($this->field_name) {
            'medical_history' => 'Medical History',
            'allergies' => 'Allergies',
            'height' => 'Height',
            'weight' => 'Weight',
            default => ucfirst(str_replace('_', ' ', $this->field_name)),
        };
    }

    /**
     * Scope for filtering by field
     */
    public function scopeForField($query, string $fieldName)
    {
        return $query->where('field_name', $fieldName);
    }

    /**
     * Scope for filtering by patient
     */
    public function scopeForPatient($query, int $patientId)
    {
        return $query->where('patient_id', $patientId);
    }

    /**
     * Scope for filtering by visit
     */
    public function scopeForVisit($query, int $visitId)
    {
        return $query->where('visit_id', $visitId);
    }
}
