<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Admission extends Model
{
    use HasFactory;

    protected $fillable = [
        'admission_number',
        'patient_id',
        'visit_id',
        'admitted_by',
        'discharged_by',
        'ward',
        'room_number',
        'bed_number',
        'admission_reason',
        'diagnosis',
        'admission_notes',
        'status',
        'admission_date',
        'discharge_date',
        'discharge_notes',
        'discharge_instructions',
    ];

    protected $casts = [
        'admission_date' => 'datetime',
        'discharge_date' => 'datetime',
    ];

    /**
     * Boot the model and generate admission number
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($admission) {
            if (empty($admission->admission_number)) {
                $admission->admission_number = static::generateAdmissionNumber();
            }
            if (empty($admission->admission_date)) {
                $admission->admission_date = now();
            }
        });
    }

    /**
     * Generate unique admission number
     */
    public static function generateAdmissionNumber(): string
    {
        $year = now()->format('Y');
        $lastAdmission = static::whereYear('created_at', $year)
            ->orderBy('id', 'desc')
            ->first();
        
        $number = $lastAdmission ? ((int) substr($lastAdmission->admission_number, -6)) + 1 : 1;
        
        return 'ADM-' . $year . '-' . str_pad($number, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Relationships
     */
    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function visit(): BelongsTo
    {
        return $this->belongsTo(Visit::class);
    }

    public function admittedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'admitted_by');
    }

    public function dischargedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'discharged_by');
    }

    public function medicationAdministrations(): HasMany
    {
        return $this->hasMany(MedicationAdministration::class);
    }

    /**
     * Check if admission is active
     */
    public function isActive(): bool
    {
        return $this->status === 'admitted';
    }

    /**
     * Discharge the patient
     */
    public function discharge(User $dischargedBy, ?string $dischargeNotes = null, ?string $dischargeInstructions = null): bool
    {
        $this->status = 'discharged';
        $this->discharged_by = $dischargedBy->id;
        $this->discharge_date = now();
        $this->discharge_notes = $dischargeNotes;
        $this->discharge_instructions = $dischargeInstructions;

        return $this->save();
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            'admitted' => 'red',
            'discharged' => 'green',
            'transferred' => 'blue',
            default => 'gray',
        };
    }

    /**
     * Get human-readable status
     */
    public function getStatusLabelAttribute(): string
    {
        return match($this->status) {
            'admitted' => 'Admitted',
            'discharged' => 'Discharged',
            'transferred' => 'Transferred',
            default => 'Unknown',
        };
    }

    /**
     * Get capitalized ward for display
     */
    public function getDisplayWardAttribute(): ?string
    {
        return $this->ward ? ucwords(strtolower($this->ward)) : null;
    }
}
