<?php

namespace App\Livewire\Visits;

use App\Models\Patient;
use App\Models\Visit;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class VisitList extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = 'active'; // 'active', 'all', 'checked_out'
    public $sortBy = 'checked_in_at';
    public $sortDirection = 'desc';
    
    // Check-in modal
    public $showCheckInModal = false;
    public $selectedPatientId = null;
    public $chiefComplaint = '';
    public $visitNotes = '';
    
    // Patient search for check-in
    public $patientSearch = '';
    public $searchResults = [];

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => 'active'],
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingStatusFilter()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function openCheckInModal()
    {
        $this->resetCheckInForm();
        $this->showCheckInModal = true;
    }

    public function closeCheckInModal()
    {
        $this->showCheckInModal = false;
        $this->resetCheckInForm();
    }

    public function resetCheckInForm()
    {
        $this->selectedPatientId = null;
        $this->chiefComplaint = '';
        $this->visitNotes = '';
        $this->patientSearch = '';
        $this->searchResults = [];
        $this->resetErrorBag();
    }

    public function searchPatients()
    {
        if (strlen($this->patientSearch) < 2) {
            $this->searchResults = [];
            return;
        }

        $searchTerm = strtolower($this->patientSearch);
        $this->searchResults = Patient::query()
            ->where(function ($query) use ($searchTerm) {
                $query->whereRaw('LOWER(first_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(last_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(patient_id) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(phone) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(email) like ?', ['%' . $searchTerm . '%']);
            })
            ->limit(10)
            ->get();
    }

    public function selectPatient($patientId)
    {
        $this->selectedPatientId = $patientId;
        $patient = Patient::find($patientId);
        $this->patientSearch = $patient->full_name . ' (' . $patient->patient_id . ')';
        $this->searchResults = [];
    }

    public function checkIn()
    {
        $rules = [
            'selectedPatientId' => 'required|exists:patients,id',
            'visitNotes' => 'nullable|string|max:2000',
        ];
        
        // Doctors, developers, and cmd can enter chief complaint
        if (auth()->user()->canEnterChiefComplaint()) {
            $rules['chiefComplaint'] = 'nullable|string|max:1000';
        }
        
        $this->validate($rules, [
            'selectedPatientId.required' => 'Please select a patient to check in.',
        ]);

        // Check if patient already has an active visit
        $patient = Patient::findOrFail($this->selectedPatientId);
        $activeVisit = $patient->active_visit;

        if ($activeVisit) {
            session()->flash('error', 'Patient already has an active visit: ' . $activeVisit->visit_number);
            return;
        }

        // Create new visit and send to nurse
        // Doctors, developers, and cmd can enter chief complaint
        $visit = Visit::create([
            'patient_id' => $this->selectedPatientId,
            'created_by' => auth()->id(),
            'status' => Visit::STATUS_WAITING_NURSE, // Immediately send to nursing station
            'chief_complaint' => auth()->user()->canEnterChiefComplaint() ? ($this->chiefComplaint ?: null) : null,
            'visit_notes' => $this->visitNotes ?: null,
        ]);

        session()->flash('message', 'Patient checked in successfully! Visit Number: ' . $visit->visit_number . ' - Patient sent to nursing station.');
        $this->closeCheckInModal();
        $this->resetPage();
    }

    public function updateStatus($visitId, $status)
    {
        $visit = Visit::findOrFail($visitId);
        
        if ($visit->updateStatus($status)) {
            session()->flash('message', 'Visit status updated successfully!');
        } else {
            session()->flash('error', 'Failed to update visit status.');
        }
    }

    public function render()
    {
        $query = Visit::with(['patient', 'createdBy'])
            ->when($this->statusFilter === 'active', function ($q) {
                $q->where('status', '!=', Visit::STATUS_CHECKED_OUT)
                  ->whereNull('checked_out_at');
            })
            ->when($this->statusFilter === 'checked_out', function ($q) {
                $q->where('status', Visit::STATUS_CHECKED_OUT)
                  ->whereNotNull('checked_out_at');
            })
            ->when($this->search, function ($q) {
                $searchTerm = strtolower($this->search);
                $q->where(function ($query) use ($searchTerm) {
                    $query->whereRaw('LOWER(visit_number) like ?', ['%' . $searchTerm . '%'])
                          ->orWhereHas('patient', function ($q) use ($searchTerm) {
                              $q->whereRaw('LOWER(first_name) like ?', ['%' . $searchTerm . '%'])
                                ->orWhereRaw('LOWER(last_name) like ?', ['%' . $searchTerm . '%'])
                                ->orWhereRaw('LOWER(patient_id) like ?', ['%' . $searchTerm . '%']);
                          });
                });
            })
            ->orderBy($this->sortBy, $this->sortDirection);

        $visits = $query->paginate(15);

        return view('livewire.visits.visit-list', [
            'visits' => $visits,
        ]);
    }
}
