<?php

namespace App\Livewire\Visits;

use App\Models\Visit;
use App\Models\Bill;
use App\Models\Payment;
use App\Models\Consultation;
use App\Models\LabRequest;
use App\Models\Prescription;
use App\Models\Service;
use App\Models\PharmacyStock;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class VisitBilling extends Component
{
    use WithPagination;

    public $selectedVisitId = null;
    public $selectedVisit = null;
    public $selectedBill = null;
    
    // Bill breakdown
    public $billItems = [];
    public $subtotal = 0;
    public $tax_rate = 0.05; // 5% tax
    public $tax_amount = 0;
    public $discount_amount = 0;
    public $total_amount = 0;
    public $insurance_covered = false;
    public $patient_responsibility = 0;
    
    // Payment
    public $showPaymentModal = false;
    public $payment_amount = 0;
    public $payment_method = 'cash';
    public $payment_reference = '';
    public $payment_notes = '';

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Check if user can manage bills (doctors should NOT have access)
        $user = auth()->user();
        if ($user->isDoctor()) {
            abort(403, 'Access denied. Doctors do not have access to billing.');
        }
        if (!$user->canManageBills()) {
            abort(403, 'Access denied.');
        }
    }

    public function selectVisit($visitId)
    {
        $this->selectedVisitId = $visitId;
        $this->selectedVisit = Visit::with(['patient', 'consultations', 'labRequests', 'prescriptions', 'bills'])->findOrFail($visitId);
        
        // Check if bill already exists
        $existingBill = Bill::where('visit_id', $visitId)->first();
        if ($existingBill) {
            $this->selectedBill = $existingBill;
            $this->loadBillData($existingBill);
        } else {
            $this->calculateBill();
        }
    }

    public function calculateBill()
    {
        $this->billItems = [];
        $this->subtotal = 0;
        $tax_rate = 0.05; // 5% tax

        // Consultation fee
        $consultation = Consultation::where('visit_id', $this->selectedVisitId)->first();
        if ($consultation) {
            $consultationService = Service::getConsultationService($consultation->consultation_type);
            if ($consultationService) {
                $consultationFee = $consultationService->price;
                $consultationName = $consultationService->name;
            } else {
                // Fallback to general consultation service
                $consultationService = Service::getConsultationService();
                $consultationFee = $consultationService ? $consultationService->price : 50.00;
                $consultationName = 'Consultation Fee';
            }
            
            $this->billItems[] = [
                'description' => $consultationName,
                'quantity' => 1,
                'unit_price' => $consultationFee,
                'total' => $consultationFee,
            ];
            $this->subtotal += $consultationFee;
        }

        // Lab test fees
        $labRequests = LabRequest::where('visit_id', $this->selectedVisitId)
            ->where('status', LabRequest::STATUS_COMPLETED)
            ->get();
        
        foreach ($labRequests as $labRequest) {
            // Try to get service by test name first, then by test type
            $labService = Service::getLabTestService($labRequest->test_name);
            
            if (!$labService && $labRequest->test_type) {
                // Try to find by test type (subcategory)
                $labService = Service::where('category', 'lab_test')
                    ->where('is_active', true)
                    ->where('subcategory', $labRequest->test_type)
                    ->first();
            }
            
            if (!$labService) {
                // Fallback to general lab test service
                $labService = Service::getLabTestService();
            }
            
            $labTestFee = $labService ? $labService->price : 25.00;
            
            $this->billItems[] = [
                'description' => 'Lab Test: ' . ucwords(strtolower($labRequest->test_name)),
                'quantity' => 1,
                'unit_price' => $labTestFee,
                'total' => $labTestFee,
            ];
            $this->subtotal += $labTestFee;
        }

        // Prescription fees - use actual drug prices from pharmacy stock
        $prescriptionService = Service::getPrescriptionService();
        $fallbackPrescriptionFee = $prescriptionService ? $prescriptionService->price : 10.00;
        
        $prescriptions = Prescription::where('visit_id', $this->selectedVisitId)
            ->with('drug.pharmacyStocks')
            ->get();
        
        foreach ($prescriptions as $prescription) {
            $drugName = $prescription->drug ? ucwords(strtolower($prescription->drug->name)) : 'Medication';
            
            // Get drug price from pharmacy stock (use latest stock entry with price)
            $drugPrice = null;
            if ($prescription->drug) {
                $pharmacyStock = PharmacyStock::where('drug_id', $prescription->drug->id)
                    ->whereNotNull('unit_price')
                    ->where('quantity', '>', 0) // Only use prices from stock that exists
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                if ($pharmacyStock && $pharmacyStock->unit_price) {
                    $drugPrice = $pharmacyStock->unit_price;
                }
            }
            
            // Use drug price if available, otherwise fallback to prescription service fee
            $unitPrice = $drugPrice ?? $fallbackPrescriptionFee;
            $prescriptionTotal = $unitPrice * $prescription->quantity;
            
            $this->billItems[] = [
                'description' => 'Prescription: ' . $drugName,
                'quantity' => $prescription->quantity,
                'unit_price' => $unitPrice,
                'total' => $prescriptionTotal,
            ];
            $this->subtotal += $prescriptionTotal;
        }

        // Calculate tax and total
        $this->tax_amount = $this->subtotal * $tax_rate;
        $this->total_amount = $this->subtotal + $this->tax_amount - $this->discount_amount;
        
        // Calculate patient responsibility
        if ($this->insurance_covered && $this->selectedVisit->patient->insurance_provider) {
            $this->patient_responsibility = $this->total_amount * 0.2; // 20% co-pay
        } else {
            $this->patient_responsibility = $this->total_amount;
        }
    }

    public function loadBillData($bill)
    {
        $this->subtotal = $bill->subtotal;
        $this->tax_amount = $bill->tax_amount;
        $this->discount_amount = $bill->discount_amount;
        $this->total_amount = $bill->total_amount;
        $this->insurance_covered = $bill->insurance_covered;
        $this->patient_responsibility = $bill->patient_responsibility;
    }

    public function updatedDiscountAmount()
    {
        $this->total_amount = $this->subtotal + $this->tax_amount - $this->discount_amount;
        if ($this->insurance_covered) {
            $this->patient_responsibility = $this->total_amount * 0.2;
        } else {
            $this->patient_responsibility = $this->total_amount;
        }
    }

    public function updatedInsuranceCovered()
    {
        if ($this->insurance_covered) {
            $this->patient_responsibility = $this->total_amount * 0.2;
        } else {
            $this->patient_responsibility = $this->total_amount;
        }
    }

    public function createBill()
    {
        if (!$this->selectedVisit) {
            session()->flash('error', 'No visit selected.');
            return;
        }

        // Check if bill already exists
        $existingBill = Bill::where('visit_id', $this->selectedVisitId)->first();
        if ($existingBill) {
            session()->flash('error', 'Bill already exists for this visit.');
            return;
        }

        $bill = Bill::create([
            'patient_id' => $this->selectedVisit->patient_id,
            'visit_id' => $this->selectedVisitId,
            'subtotal' => $this->subtotal,
            'tax_amount' => $this->tax_amount,
            'discount_amount' => $this->discount_amount,
            'total_amount' => $this->total_amount,
            'due_date' => now()->addDays(30),
            'status' => 'pending',
            'created_by' => auth()->id(),
            'billing_period_start' => $this->selectedVisit->checked_in_at->toDateString(),
            'billing_period_end' => $this->selectedVisit->checked_in_at->toDateString(),
            'insurance_covered' => $this->insurance_covered,
            'patient_responsibility' => $this->patient_responsibility,
        ]);

        $this->selectedBill = $bill;
        $this->selectedVisit->updateStatus(Visit::STATUS_BILLING_PENDING);
        
        session()->flash('message', 'Bill created successfully!');
    }

    public function openPaymentModal()
    {
        if (!$this->selectedBill) {
            session()->flash('error', 'Please create a bill first.');
            return;
        }
        $this->payment_amount = $this->selectedBill->balance;
        $this->showPaymentModal = true;
    }

    public function closePaymentModal()
    {
        $this->showPaymentModal = false;
        $this->payment_amount = 0;
        $this->payment_method = 'cash';
        $this->payment_reference = '';
        $this->payment_notes = '';
    }

    public function processPayment()
    {
        $this->validate([
            'payment_amount' => 'required|numeric|min:0.01|max:' . $this->selectedBill->balance,
            'payment_method' => 'required|in:cash,card,hmo,other',
        ]);

        $payment = Payment::create([
            'bill_id' => $this->selectedBill->id,
            'amount' => $this->payment_amount,
            'payment_method' => $this->payment_method,
            'payment_reference' => $this->payment_reference ?: null,
            'notes' => $this->payment_notes ?: null,
            'processed_by' => auth()->id(),
            'status' => 'completed',
            'payment_date' => now(),
        ]);

        // Update bill status
        $this->selectedBill->refresh();
        if ($this->selectedBill->balance <= 0) {
            $this->selectedBill->status = 'paid';
            $this->selectedBill->save();
        } else {
            $this->selectedBill->status = 'partial';
            $this->selectedBill->save();
        }

        session()->flash('message', 'Payment processed successfully!');
        $this->closePaymentModal();
    }

    public function checkout()
    {
        if (!$this->selectedBill) {
            session()->flash('error', 'Please create and pay the bill first.');
            return;
        }

        if ($this->selectedBill->balance > 0) {
            session()->flash('error', 'Bill must be fully paid before checkout.');
            return;
        }

        // Update visit status to checked out
        $this->selectedVisit->updateStatus(Visit::STATUS_CHECKED_OUT);
        
        session()->flash('message', 'Patient checked out successfully! Visit completed.');
        $this->selectedVisit = null;
        $this->selectedBill = null;
        $this->selectedVisitId = null;
        $this->resetPage();
    }

    public function render()
    {
        // Get visits ready for billing (consultation completed, not checked out)
        $visits = Visit::with(['patient', 'consultations', 'bills'])
            ->whereIn('status', [
                Visit::STATUS_IN_CONSULTATION,
                Visit::STATUS_LAB_REQUESTED,
                Visit::STATUS_PRESCRIPTION_ISSUED,
                Visit::STATUS_PHARMACY_DISPENSED,
                Visit::STATUS_BILLING_PENDING,
            ])
            ->whereNull('checked_out_at')
            ->orderBy('checked_in_at', 'desc')
            ->paginate(15);

        return view('livewire.visits.visit-billing', [
            'visits' => $visits,
        ]);
    }
}
