<?php

namespace App\Livewire\Surgery;

use App\Models\Operation;
use App\Models\OperationRoom;
use App\Models\Patient;
use App\Models\User;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class SurgeryList extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = '';
    public $dateFilter = '';
    public $showCreateModal = false;
    public $showEditModal = false;
    public $showViewModal = false;
    public $editingOperation = null;
    public $selectedOperation = null;

    // Form fields
    public $patient_id = '';
    public $visit_id = '';
    public $surgeon_id = '';
    public $assistant_surgeon_id = '';
    public $anesthesiologist_id = '';
    public $operation_room_id = '';
    public $surgery_type = '';
    public $procedure_name = '';
    public $diagnosis = '';
    public $indication = '';
    public $scheduled_date = '';
    public $scheduled_time = '';
    public $estimated_duration = 60;
    public $status = 'scheduled';
    public $pre_operative_notes = '';
    public $operative_notes = '';
    public $post_operative_notes = '';
    public $complications = '';
    public $notes = '';
    
    // Patient search
    public $patientSearch = '';
    public $searchResults = [];

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'dateFilter' => ['except' => ''],
    ];

    public function mount()
    {
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->surgeon_id = auth()->id();
        $this->scheduled_date = now()->format('Y-m-d');
        $this->scheduled_time = now()->addHours(2)->format('H:i');
        $this->showCreateModal = true;
    }

    public function openViewModal($operationId)
    {
        $this->selectedOperation = Operation::with([
            'patient', 'visit', 'surgeon', 'assistantSurgeon', 
            'anesthesiologist', 'operationRoom', 'bookedBy'
        ])->findOrFail($operationId);
        $this->showViewModal = true;
    }

    public function openEditModal($operationId)
    {
        $this->showViewModal = false;
        $operation = Operation::findOrFail($operationId);
        $this->editingOperation = $operation;
        $this->patient_id = $operation->patient_id;
        if ($this->patient_id) {
            $patient = Patient::find($this->patient_id);
            if ($patient) {
                $this->patientSearch = $patient->patient_id . ' - ' . ucwords(strtolower($patient->full_name));
            }
        }
        $this->visit_id = $operation->visit_id ?? '';
        $this->surgeon_id = $operation->surgeon_id;
        $this->assistant_surgeon_id = $operation->assistant_surgeon_id ?? '';
        $this->anesthesiologist_id = $operation->anesthesiologist_id ?? '';
        $this->operation_room_id = $operation->operation_room_id ?? '';
        $this->surgery_type = $operation->surgery_type;
        $this->procedure_name = $operation->procedure_name;
        $this->diagnosis = $operation->diagnosis ?? '';
        $this->indication = $operation->indication ?? '';
        $this->scheduled_date = $operation->scheduled_date->format('Y-m-d');
        $this->scheduled_time = $operation->scheduled_time;
        $this->estimated_duration = $operation->estimated_duration;
        $this->status = $operation->status;
        $this->pre_operative_notes = $operation->pre_operative_notes ?? '';
        $this->operative_notes = $operation->operative_notes ?? '';
        $this->post_operative_notes = $operation->post_operative_notes ?? '';
        $this->complications = $operation->complications ?? '';
        $this->notes = $operation->notes ?? '';
        $this->showEditModal = true;
    }

    public function closeModals()
    {
        $this->showCreateModal = false;
        $this->showEditModal = false;
        $this->showViewModal = false;
        $this->editingOperation = null;
        $this->selectedOperation = null;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->patient_id = '';
        $this->visit_id = '';
        $this->surgeon_id = '';
        $this->assistant_surgeon_id = '';
        $this->anesthesiologist_id = '';
        $this->operation_room_id = '';
        $this->surgery_type = '';
        $this->procedure_name = '';
        $this->diagnosis = '';
        $this->indication = '';
        $this->scheduled_date = '';
        $this->scheduled_time = '';
        $this->estimated_duration = 60;
        $this->status = 'scheduled';
        $this->pre_operative_notes = '';
        $this->operative_notes = '';
        $this->post_operative_notes = '';
        $this->complications = '';
        $this->notes = '';
        $this->patientSearch = '';
        $this->searchResults = [];
        $this->resetErrorBag();
    }
    
    public function searchPatients()
    {
        if (strlen($this->patientSearch) < 2) {
            $this->searchResults = [];
            return;
        }

        $searchTerm = strtolower($this->patientSearch);
        $this->searchResults = Patient::query()
            ->where(function ($query) use ($searchTerm) {
                $query->whereRaw('LOWER(first_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(last_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(patient_id) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(phone) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(email) like ?', ['%' . $searchTerm . '%']);
            })
            ->limit(10)
            ->get();
    }

    public function selectPatient($patientId)
    {
        $this->patient_id = $patientId;
        $patient = Patient::find($patientId);
        if ($patient) {
            $this->patientSearch = $patient->patient_id . ' - ' . ucwords(strtolower($patient->full_name));
        }
        $this->searchResults = [];
    }

    public function save()
    {
        $rules = [
            'patient_id' => 'required|exists:patients,id',
            'surgeon_id' => 'required|exists:users,id',
            'surgery_type' => 'required|string|max:255',
            'procedure_name' => 'required|string|max:500',
            'scheduled_date' => 'required|date|after_or_equal:today',
            'scheduled_time' => 'required',
            'estimated_duration' => 'required|integer|min:15|max:1440',
            'status' => 'required|in:scheduled,preparing,in_progress,completed,cancelled,postponed',
        ];

        if ($this->operation_room_id) {
            $rules['operation_room_id'] = 'exists:operation_rooms,id';
            
            // Check room availability
            $room = OperationRoom::find($this->operation_room_id);
            if ($room && !$room->isAvailableFor(
                $this->scheduled_date, 
                $this->scheduled_time, 
                $this->estimated_duration,
                $this->editingOperation?->id
            )) {
                session()->flash('error', 'Operation room is not available at the selected date and time.');
                return;
            }
        }

        $this->validate($rules);

        $data = [
            'patient_id' => $this->patient_id,
            'visit_id' => $this->visit_id ?: null,
            'surgeon_id' => $this->surgeon_id,
            'assistant_surgeon_id' => $this->assistant_surgeon_id ?: null,
            'anesthesiologist_id' => $this->anesthesiologist_id ?: null,
            'operation_room_id' => $this->operation_room_id ?: null,
            'surgery_type' => ucwords(strtolower($this->surgery_type)),
            'procedure_name' => ucwords(strtolower($this->procedure_name)),
            'diagnosis' => $this->diagnosis ?: null,
            'indication' => $this->indication ?: null,
            'scheduled_date' => $this->scheduled_date,
            'scheduled_time' => $this->scheduled_time,
            'estimated_duration' => $this->estimated_duration,
            'status' => $this->status,
            'pre_operative_notes' => $this->pre_operative_notes ?: null,
            'operative_notes' => $this->operative_notes ?: null,
            'post_operative_notes' => $this->post_operative_notes ?: null,
            'complications' => $this->complications ?: null,
            'notes' => $this->notes ?: null,
            'booked_by' => auth()->id(),
        ];

        if ($this->editingOperation) {
            $this->editingOperation->update($data);
            session()->flash('message', 'Surgery updated successfully!');
        } else {
            Operation::create($data);
            session()->flash('message', 'Surgery booked successfully!');
        }

        $this->closeModals();
    }

    public function startOperation($operationId)
    {
        $operation = Operation::findOrFail($operationId);
        if ($operation->canStart()) {
            $operation->start();
            session()->flash('message', 'Operation started successfully!');
        } else {
            session()->flash('error', 'Operation cannot be started in its current status.');
        }
    }

    public function completeOperation($operationId)
    {
        $operation = Operation::findOrFail($operationId);
        if ($operation->canComplete()) {
            $operation->complete();
            session()->flash('message', 'Operation completed successfully!');
        } else {
            session()->flash('error', 'Operation cannot be completed in its current status.');
        }
    }

    public function cancelOperation($operationId)
    {
        $operation = Operation::findOrFail($operationId);
        if (in_array($operation->status, [Operation::STATUS_SCHEDULED, Operation::STATUS_PREPARING])) {
            $operation->update(['status' => Operation::STATUS_CANCELLED]);
            
            // Free up the operation room
            if ($operation->operationRoom) {
                $operation->operationRoom->update(['status' => OperationRoom::STATUS_AVAILABLE]);
            }
            
            session()->flash('message', 'Operation cancelled successfully!');
        } else {
            session()->flash('error', 'Operation cannot be cancelled in its current status.');
        }
    }

    public function render()
    {
        $operations = Operation::query()
            ->with(['patient', 'surgeon', 'operationRoom'])
            ->when($this->search, function ($query) {
                $searchTerm = strtolower($this->search);
                $query->where(function ($q) use ($searchTerm) {
                    $q->whereRaw('LOWER(operation_number) LIKE ?', ['%' . $searchTerm . '%'])
                      ->orWhereHas('patient', function ($q) use ($searchTerm) {
                          $q->whereRaw('LOWER(first_name) LIKE ?', ['%' . $searchTerm . '%'])
                            ->orWhereRaw('LOWER(last_name) LIKE ?', ['%' . $searchTerm . '%'])
                            ->orWhereRaw('LOWER(patient_id) LIKE ?', ['%' . $searchTerm . '%']);
                      })
                      ->orWhereRaw('LOWER(procedure_name) LIKE ?', ['%' . $searchTerm . '%']);
                });
            })
            ->when($this->statusFilter, function ($query) {
                $query->where('status', $this->statusFilter);
            })
            ->when($this->dateFilter, function ($query) {
                if ($this->dateFilter === 'today') {
                    $query->whereDate('scheduled_date', today());
                } elseif ($this->dateFilter === 'upcoming') {
                    $query->whereDate('scheduled_date', '>=', today());
                } elseif ($this->dateFilter === 'past') {
                    $query->whereDate('scheduled_date', '<', today());
                }
            })
            ->orderBy('scheduled_date', 'desc')
            ->orderBy('scheduled_time', 'desc')
            ->paginate(15);

        $patients = Patient::orderBy('first_name')->get();
        $doctors = User::where(function($query) {
            $query->where('role', 'doctor')
                  ->orWhere('role', 'staff')
                  ->orWhereNull('role');
        })->orderBy('name')->get();
        $operationRooms = OperationRoom::where('is_active', true)->orderBy('room_number')->get();

        return view('livewire.surgery.surgery-list', [
            'operations' => $operations,
            'patients' => $patients,
            'doctors' => $doctors,
            'operationRooms' => $operationRooms,
        ]);
    }
}
