<?php

namespace App\Livewire\Pharmacy;

use App\Models\PharmacyStock;
use App\Models\Drug;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;

#[Layout('layouts.app')]
class InventoryManagement extends Component
{
    use WithPagination, WithFileUploads;

    public $search = '';
    public $statusFilter = '';
    
    // Form fields
    public $showModal = false;
    public $showBulkUploadModal = false;
    public $editingStock = null;
    
    // Bulk upload fields
    public $bulk_upload_file;
    public $drug_id = '';
    public $quantity = 0;
    public $reorder_level = 10;
    public $unit_price = '';
    public $expiry_date = '';
    public $batch_number = '';
    public $supplier = '';
    public $supply_date = '';
    public $notes = '';

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => ''],
    ];

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Only pharmacy, developer, and cmd can manage inventory
        $user = auth()->user();
        if (!$user->isPharmacy() && !$user->isDeveloper() && !$user->isCmd()) {
            abort(403, 'Access denied. This page is for pharmacy staff, developers, and CMD only.');
        }
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->supply_date = now()->format('Y-m-d');
        $this->showModal = true;
    }

    public function openEditModal($stockId)
    {
        $stock = PharmacyStock::with('drug')->findOrFail($stockId);
        $this->editingStock = $stock;
        $this->drug_id = $stock->drug_id;
        $this->quantity = $stock->quantity;
        $this->reorder_level = $stock->reorder_level;
        $this->unit_price = $stock->unit_price ?? '';
        $this->expiry_date = $stock->expiry_date ? $stock->expiry_date->format('Y-m-d') : '';
        $this->batch_number = $stock->batch_number ?? '';
        $this->supplier = $stock->supplier ?? '';
        $this->supply_date = $stock->supply_date ? $stock->supply_date->format('Y-m-d') : now()->format('Y-m-d');
        $this->notes = $stock->notes ?? '';
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->editingStock = null;
        $this->drug_id = '';
        $this->quantity = 0;
        $this->reorder_level = 10;
        $this->unit_price = '';
        $this->expiry_date = '';
        $this->batch_number = '';
        $this->supplier = '';
        $this->supply_date = '';
        $this->notes = '';
        $this->resetErrorBag();
    }

    public function save()
    {
        $rules = [
            'drug_id' => 'required|exists:drugs,id',
            'quantity' => 'required|integer|min:0',
            'reorder_level' => 'required|integer|min:0',
            'unit_price' => 'nullable|numeric|min:0',
            'expiry_date' => 'nullable|date',
            'batch_number' => 'nullable|string|max:255',
            'supplier' => 'nullable|string|max:255',
            'supply_date' => 'nullable|date',
            'notes' => 'nullable|string',
        ];

        $this->validate($rules);

        $data = [
            'drug_id' => $this->drug_id,
            'quantity' => $this->quantity,
            'reorder_level' => $this->reorder_level,
            'unit_price' => $this->unit_price ?: null,
            'expiry_date' => $this->expiry_date ?: null,
            'batch_number' => $this->batch_number ?: null,
            'supplier' => $this->supplier ?: null,
            'supply_date' => $this->supply_date ?: null,
            'notes' => $this->notes ?: null,
        ];

        if ($this->editingStock) {
            $this->editingStock->update($data);
            session()->flash('message', 'Stock updated successfully!');
        } else {
            PharmacyStock::create($data);
            session()->flash('message', 'Stock entry created successfully!');
        }

        $this->closeModal();
    }

    public function delete($stockId)
    {
        $stock = PharmacyStock::with('drug')->findOrFail($stockId);
        $drugName = $stock->drug->name;
        $stock->delete();
        
        session()->flash('message', "Stock entry for '{$drugName}' deleted successfully!");
    }

    // Bulk Upload Methods
    public function openBulkUploadModal()
    {
        $this->bulk_upload_file = null;
        $this->resetErrorBag();
        $this->showBulkUploadModal = true;
    }

    public function closeBulkUploadModal()
    {
        $this->showBulkUploadModal = false;
        $this->bulk_upload_file = null;
        $this->resetErrorBag();
    }

    public function processBulkUpload()
    {
        $this->validate([
            'bulk_upload_file' => 'required|mimes:csv,txt|max:5120', // 5MB max
        ]);

        try {
            $path = $this->bulk_upload_file->storeAs('temp', 'bulk_inventory_upload_' . time() . '.csv');
            $fullPath = storage_path('app/' . $path);

            $file = fopen($fullPath, 'r');
            
            // Skip header row if present
            $header = fgetcsv($file);
            
            $successCount = 0;
            $errorCount = 0;
            $errors = [];

            while (($row = fgetcsv($file)) !== FALSE) {
                if (count($row) < 3) { // Minimum required fields: drug_name, quantity, unit_price
                    $errorCount++;
                    continue;
                }

                try {
                    // Map CSV columns: drug_name, quantity, reorder_level, unit_price, expiry_date, batch_number, supplier, supply_date, notes
                    $drugName = trim($row[0] ?? '');
                    $quantity = isset($row[1]) ? (int)trim($row[1]) : 0;
                    $reorderLevel = isset($row[2]) ? (int)trim($row[2]) : 10;
                    $unitPrice = isset($row[3]) && !empty(trim($row[3])) ? (float)trim($row[3]) : null;
                    $expiryDate = isset($row[4]) && !empty(trim($row[4])) ? trim($row[4]) : null;
                    $batchNumber = isset($row[5]) && !empty(trim($row[5])) ? trim($row[5]) : null;
                    $supplier = isset($row[6]) && !empty(trim($row[6])) ? trim($row[6]) : null;
                    $supplyDate = isset($row[7]) && !empty(trim($row[7])) ? trim($row[7]) : now()->format('Y-m-d');
                    $notes = isset($row[8]) && !empty(trim($row[8])) ? trim($row[8]) : null;

                    // Validate required fields
                    if (empty($drugName) || $quantity < 0) {
                        $errorCount++;
                        continue;
                    }

                    // Find drug by name
                    $drug = Drug::where('name', $drugName)->first();
                    if (!$drug) {
                        $errorCount++;
                        $errors[] = "Drug '{$drugName}' not found. Please create the drug first.";
                        continue;
                    }

                    // Validate dates
                    if ($expiryDate && !strtotime($expiryDate)) {
                        $expiryDate = null;
                    }
                    if ($supplyDate && !strtotime($supplyDate)) {
                        $supplyDate = now()->format('Y-m-d');
                    }

                    // Create inventory entry
                    PharmacyStock::create([
                        'drug_id' => $drug->id,
                        'quantity' => $quantity,
                        'reorder_level' => $reorderLevel,
                        'unit_price' => $unitPrice,
                        'expiry_date' => $expiryDate,
                        'batch_number' => $batchNumber,
                        'supplier' => $supplier,
                        'supply_date' => $supplyDate,
                        'notes' => $notes,
                    ]);

                    $successCount++;
                } catch (\Exception $e) {
                    $errorCount++;
                    $errors[] = "Row error for '{$drugName}': " . $e->getMessage();
                }
            }

            fclose($file);
            unlink($fullPath); // Clean up temp file

            $message = "Bulk upload completed! Successfully processed {$successCount} inventory entry/entries";
            if ($errorCount > 0) {
                $message .= ", {$errorCount} error(s) encountered.";
            }
            
            session()->flash('message', $message);
            if (!empty($errors)) {
                session()->flash('bulk_upload_errors', array_slice($errors, 0, 10)); // Show first 10 errors
            }

            $this->closeBulkUploadModal();
        } catch (\Exception $e) {
            session()->flash('error', 'Error processing bulk upload: ' . $e->getMessage());
            $this->closeBulkUploadModal();
        }
    }

    public function render()
    {
        $stocks = PharmacyStock::with('drug')
            ->when($this->search, function ($query) {
                $query->whereHas('drug', function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('generic_name', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->statusFilter === 'low', function ($query) {
                $query->lowStock();
            })
            ->when($this->statusFilter === 'expiring', function ($query) {
                $query->expiringSoon(30);
            })
            ->when($this->statusFilter === 'expired', function ($query) {
                $query->expired();
            })
            ->orderBy('drug_id')
            ->orderBy('expiry_date', 'asc')
            ->paginate(15);

        $drugs = Drug::active()->orderBy('name')->get();

        return view('livewire.pharmacy.inventory-management', [
            'stocks' => $stocks,
            'drugs' => $drugs,
        ]);
    }
}
