<?php

namespace App\Livewire\Pharmacy;

use App\Models\Drug;
use App\Models\DrugCategory;
use App\Models\PharmacyStock;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;
use Illuminate\Support\Str;

#[Layout('layouts.app')]
class DrugManagement extends Component
{
    use WithPagination, WithFileUploads;

    public $search = '';
    public $categoryFilter = '';
    public $statusFilter = '';
    
    // Form fields
    public $showModal = false;
    public $showCategoryModal = false;
    public $showBulkUploadModal = false;
    public $editingDrug = null;
    
    // Category form fields
    public $category_name = '';
    public $category_description = '';
    
    // Bulk upload fields
    public $bulk_upload_file;
    public $name = '';
    public $generic_name = '';
    public $manufacturer = '';
    public $category = '';
    public $category_id = '';
    public $description = '';
    public $dosage_form = '';
    public $strength = '';
    public $unit = '';
    public $prescription_required = true;
    public $side_effects = [];
    public $side_effect_input = '';
    public $contraindications = [];
    public $contraindication_input = '';
    public $interactions = [];
    public $interaction_input = '';
    public $storage_conditions = '';
    public $expiry_alert_days = 30;
    public $is_active = true;

    protected $queryString = [
        'search' => ['except' => ''],
        'categoryFilter' => ['except' => ''],
        'statusFilter' => ['except' => ''],
    ];

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Only pharmacy, developer, and cmd can manage drugs
        $user = auth()->user();
        if (!$user->isPharmacy() && !$user->isDeveloper() && !$user->isCmd()) {
            abort(403, 'Access denied. This page is for pharmacy staff, developers, and CMD only.');
        }
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    public function openEditModal($drugId)
    {
        $drug = Drug::with('drugCategory')->findOrFail($drugId);
        $this->editingDrug = $drug;
        $this->name = $drug->name;
        $this->generic_name = $drug->generic_name;
        $this->manufacturer = $drug->manufacturer;
        $this->category = $drug->category;
        $this->category_id = $drug->category_id ?? '';
        $this->description = $drug->description ?? '';
        $this->dosage_form = $drug->dosage_form;
        $this->strength = $drug->strength;
        $this->unit = $drug->unit;
        $this->prescription_required = $drug->prescription_required;
        $this->side_effects = $drug->side_effects ?? [];
        $this->contraindications = $drug->contraindications ?? [];
        $this->interactions = $drug->interactions ?? [];
        $this->storage_conditions = $drug->storage_conditions ?? '';
        $this->expiry_alert_days = $drug->expiry_alert_days;
        $this->is_active = $drug->is_active;
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->editingDrug = null;
        $this->name = '';
        $this->generic_name = '';
        $this->manufacturer = '';
        $this->category = '';
        $this->category_id = '';
        $this->description = '';
        $this->dosage_form = '';
        $this->strength = '';
        $this->unit = '';
        $this->prescription_required = true;
        $this->side_effects = [];
        $this->side_effect_input = '';
        $this->contraindications = [];
        $this->contraindication_input = '';
        $this->interactions = [];
        $this->interaction_input = '';
        $this->storage_conditions = '';
        $this->expiry_alert_days = 30;
        $this->is_active = true;
        $this->resetErrorBag();
    }

    public function addSideEffect()
    {
        if (!empty($this->side_effect_input)) {
            $this->side_effects[] = $this->side_effect_input;
            $this->side_effect_input = '';
        }
    }

    public function removeSideEffect($index)
    {
        unset($this->side_effects[$index]);
        $this->side_effects = array_values($this->side_effects);
    }

    public function addContraindication()
    {
        if (!empty($this->contraindication_input)) {
            $this->contraindications[] = $this->contraindication_input;
            $this->contraindication_input = '';
        }
    }

    public function removeContraindication($index)
    {
        unset($this->contraindications[$index]);
        $this->contraindications = array_values($this->contraindications);
    }

    public function addInteraction()
    {
        if (!empty($this->interaction_input)) {
            $this->interactions[] = $this->interaction_input;
            $this->interaction_input = '';
        }
    }

    public function removeInteraction($index)
    {
        unset($this->interactions[$index]);
        $this->interactions = array_values($this->interactions);
    }

    public function save()
    {
        $rules = [
            'name' => 'required|string|max:255',
            'generic_name' => 'required|string|max:255',
            'manufacturer' => 'required|string|max:255',
            'category_id' => 'required|exists:drug_categories,id',
            'dosage_form' => 'required|string|max:255',
            'strength' => 'required|string|max:255',
            'unit' => 'required|string|max:255',
            'prescription_required' => 'boolean',
            'expiry_alert_days' => 'integer|min:0',
            'is_active' => 'boolean',
        ];

        $this->validate($rules);

        // Get category name if category_id is set
        $categoryName = null;
        if ($this->category_id) {
            $category = DrugCategory::find($this->category_id);
            $categoryName = $category ? $category->name : null;
        }

        $data = [
            'name' => $this->name,
            'generic_name' => $this->generic_name,
            'manufacturer' => $this->manufacturer,
            'category_id' => $this->category_id,
            'category' => $categoryName, // Set category name for backward compatibility
            'description' => $this->description ?: null,
            'dosage_form' => $this->dosage_form,
            'strength' => $this->strength,
            'unit' => $this->unit,
            'prescription_required' => $this->prescription_required,
            'side_effects' => !empty($this->side_effects) ? $this->side_effects : null,
            'contraindications' => !empty($this->contraindications) ? $this->contraindications : null,
            'interactions' => !empty($this->interactions) ? $this->interactions : null,
            'storage_conditions' => $this->storage_conditions ?: null,
            'expiry_alert_days' => $this->expiry_alert_days,
            'is_active' => $this->is_active,
        ];

        if ($this->editingDrug) {
            $this->editingDrug->update($data);
            session()->flash('message', 'Drug updated successfully!');
        } else {
            Drug::create($data);
            session()->flash('message', 'Drug created successfully!');
        }

        $this->closeModal();
    }

    public function delete($drugId)
    {
        $drug = Drug::findOrFail($drugId);
        $drugName = $drug->name;
        
        // Check if drug is used in prescriptions
        if ($drug->prescriptions()->count() > 0) {
            session()->flash('error', "Cannot delete drug '{$drugName}' because it is used in existing prescriptions.");
            return;
        }
        
        $drug->delete();
        session()->flash('message', "Drug '{$drugName}' deleted successfully!");
    }

    public function toggleStatus($drugId)
    {
        $drug = Drug::findOrFail($drugId);
        $drug->update(['is_active' => !$drug->is_active]);
        
        $status = $drug->is_active ? 'activated' : 'deactivated';
        session()->flash('message', "Drug '{$drug->name}' {$status} successfully!");
    }

    // Category Management Methods
    public function openCategoryModal()
    {
        $this->category_name = '';
        $this->category_description = '';
        $this->resetErrorBag();
        $this->showCategoryModal = true;
    }

    public function closeCategoryModal()
    {
        $this->showCategoryModal = false;
        $this->category_name = '';
        $this->category_description = '';
        $this->resetErrorBag();
    }

    public function saveCategory()
    {
        $this->validate([
            'category_name' => 'required|string|max:255|unique:drug_categories,name',
            'category_description' => 'nullable|string',
        ], [
            'category_name.unique' => 'A category with this name already exists.',
        ]);

        DrugCategory::create([
            'name' => $this->category_name,
            'slug' => Str::slug($this->category_name),
            'description' => $this->category_description ?: null,
            'is_active' => true,
            'sort_order' => DrugCategory::max('sort_order') + 1,
        ]);

        session()->flash('message', "Category '{$this->category_name}' created successfully!");
        $this->closeCategoryModal();
    }

    // Bulk Upload Methods
    public function openBulkUploadModal()
    {
        $this->bulk_upload_file = null;
        $this->resetErrorBag();
        $this->showBulkUploadModal = true;
    }

    public function closeBulkUploadModal()
    {
        $this->showBulkUploadModal = false;
        $this->bulk_upload_file = null;
        $this->resetErrorBag();
    }

    public function processBulkUpload()
    {
        $this->validate([
            'bulk_upload_file' => 'required|mimes:csv,txt|max:5120', // 5MB max
        ]);

        try {
            $path = $this->bulk_upload_file->storeAs('temp', 'bulk_upload_' . time() . '.csv');
            $fullPath = storage_path('app/' . $path);

            $file = fopen($fullPath, 'r');
            
            // Skip header row if present
            $header = fgetcsv($file);
            $expectedHeaders = ['name', 'generic_name', 'manufacturer', 'category', 'dosage_form', 'strength', 'unit', 'description', 'prescription_required', 'is_active'];
            
            $successCount = 0;
            $errorCount = 0;
            $errors = [];

            while (($row = fgetcsv($file)) !== FALSE) {
                if (count($row) < 7) { // Minimum required fields
                    $errorCount++;
                    continue;
                }

                try {
                    // Map CSV columns: name, generic_name, manufacturer, category, dosage_form, strength, unit, description, prescription_required, is_active
                    // Optional inventory columns: quantity, reorder_level, unit_price, expiry_date, batch_number, supplier, supply_date, notes
                    $name = trim($row[0] ?? '');
                    $genericName = trim($row[1] ?? '');
                    $manufacturer = trim($row[2] ?? '');
                    $categoryName = trim($row[3] ?? '');
                    $dosageForm = trim($row[4] ?? '');
                    $strength = trim($row[5] ?? '');
                    $unit = trim($row[6] ?? '');
                    $description = trim($row[7] ?? '');
                    $prescriptionRequired = isset($row[8]) ? filter_var(trim($row[8]), FILTER_VALIDATE_BOOLEAN) : true;
                    $isActive = isset($row[9]) ? filter_var(trim($row[9]), FILTER_VALIDATE_BOOLEAN) : true;
                    
                    // Inventory fields (optional columns 10-17)
                    $quantity = isset($row[10]) && !empty(trim($row[10])) ? (int)trim($row[10]) : null;
                    $reorderLevel = isset($row[11]) && !empty(trim($row[11])) ? (int)trim($row[11]) : null;
                    $unitPrice = isset($row[12]) && !empty(trim($row[12])) ? (float)trim($row[12]) : null;
                    $expiryDate = isset($row[13]) && !empty(trim($row[13])) ? trim($row[13]) : null;
                    $batchNumber = isset($row[14]) && !empty(trim($row[14])) ? trim($row[14]) : null;
                    $supplier = isset($row[15]) && !empty(trim($row[15])) ? trim($row[15]) : null;
                    $supplyDate = isset($row[16]) && !empty(trim($row[16])) ? trim($row[16]) : null;
                    $notes = isset($row[17]) && !empty(trim($row[17])) ? trim($row[17]) : null;

                    // Validate required fields
                    if (empty($name) || empty($genericName) || empty($manufacturer) || empty($categoryName) || empty($dosageForm) || empty($strength) || empty($unit)) {
                        $errorCount++;
                        continue;
                    }

                    // Get or create category
                    $category = DrugCategory::where('name', $categoryName)->first();
                    if (!$category) {
                        $category = DrugCategory::create([
                            'name' => $categoryName,
                            'slug' => Str::slug($categoryName),
                            'is_active' => true,
                            'sort_order' => DrugCategory::max('sort_order') + 1,
                        ]);
                    }

                    // Check if drug already exists
                    $existingDrug = Drug::where('name', $name)
                        ->where('generic_name', $genericName)
                        ->where('dosage_form', $dosageForm)
                        ->where('strength', $strength)
                        ->first();

                    if ($existingDrug) {
                        // Update existing drug
                        $existingDrug->update([
                            'manufacturer' => $manufacturer,
                            'category_id' => $category->id,
                            'category' => $category->name,
                            'unit' => $unit,
                            'description' => $description ?: null,
                            'prescription_required' => $prescriptionRequired,
                            'is_active' => $isActive,
                        ]);
                        $drug = $existingDrug;
                    } else {
                        // Create new drug
                        $drug = Drug::create([
                            'name' => $name,
                            'generic_name' => $genericName,
                            'manufacturer' => $manufacturer,
                            'category_id' => $category->id,
                            'category' => $category->name,
                            'dosage_form' => $dosageForm,
                            'strength' => $strength,
                            'unit' => $unit,
                            'description' => $description ?: null,
                            'prescription_required' => $prescriptionRequired,
                            'is_active' => $isActive,
                        ]);
                    }

                    // Create inventory entry if quantity is provided
                    if ($quantity !== null && $quantity >= 0) {
                        // Validate dates
                        $expiryDateFormatted = null;
                        if ($expiryDate && strtotime($expiryDate)) {
                            $expiryDateFormatted = date('Y-m-d', strtotime($expiryDate));
                        }
                        
                        $supplyDateFormatted = $supplyDate && strtotime($supplyDate) 
                            ? date('Y-m-d', strtotime($supplyDate)) 
                            : now()->format('Y-m-d');

                        PharmacyStock::create([
                            'drug_id' => $drug->id,
                            'quantity' => $quantity,
                            'reorder_level' => $reorderLevel ?? 10,
                            'unit_price' => $unitPrice,
                            'expiry_date' => $expiryDateFormatted,
                            'batch_number' => $batchNumber,
                            'supplier' => $supplier,
                            'supply_date' => $supplyDateFormatted,
                            'notes' => $notes,
                        ]);
                    }

                    $successCount++;
                } catch (\Exception $e) {
                    $errorCount++;
                    $errors[] = "Row error: " . $e->getMessage();
                }
            }

            fclose($file);
            unlink($fullPath); // Clean up temp file

            $message = "Bulk upload completed! Successfully processed {$successCount} drug(s)";
            if ($errorCount > 0) {
                $message .= ", {$errorCount} error(s) encountered.";
            }
            
            session()->flash('message', $message);
            if (!empty($errors)) {
                session()->flash('bulk_upload_errors', array_slice($errors, 0, 10)); // Show first 10 errors
            }

            $this->closeBulkUploadModal();
        } catch (\Exception $e) {
            session()->flash('error', 'Error processing bulk upload: ' . $e->getMessage());
            $this->closeBulkUploadModal();
        }
    }

    public function render()
    {
        $drugs = Drug::query()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('generic_name', 'like', '%' . $this->search . '%')
                      ->orWhere('manufacturer', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->categoryFilter, function ($query) {
                if (is_numeric($this->categoryFilter)) {
                    $query->where('category_id', $this->categoryFilter);
                } else {
                    $query->where('category', $this->categoryFilter);
                }
            })
            ->when($this->statusFilter === 'active', function ($query) {
                $query->where('is_active', true);
            })
            ->when($this->statusFilter === 'inactive', function ($query) {
                $query->where('is_active', false);
            })
            ->with(['drugCategory', 'pharmacyStocks'])
            ->orderBy('category_id')
            ->orderBy('category')
            ->orderBy('name')
            ->paginate(15);

        $drugCategories = DrugCategory::active()->ordered()->get();
        $dosageForms = ['Tablet', 'Capsule', 'Syrup', 'Injection', 'Cream', 'Ointment', 'Drops', 'Inhaler', 'Spray', 'Other'];

        return view('livewire.pharmacy.drug-management', [
            'drugs' => $drugs,
            'drugCategories' => $drugCategories,
            'dosageForms' => $dosageForms,
        ]);
    }
}
