<?php

namespace App\Livewire\Pharmacy;

use App\Models\Drug;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('layouts.app')]
class DrugDetail extends Component
{
    public Drug $drug;

    public function mount($id)
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Only pharmacy, developer, and cmd can view drug details
        $user = auth()->user();
        if (!$user->isPharmacy() && !$user->isDeveloper() && !$user->isCmd()) {
            abort(403, 'Access denied. This page is for pharmacy staff, developers, and CMD only.');
        }

        $this->drug = Drug::with([
            'drugCategory',
            'pharmacyStocks' => function($query) {
                $query->orderBy('expiry_date', 'asc')->orderBy('created_at', 'desc');
            },
            'prescriptions' => function($query) {
                $query->latest()->limit(10);
            },
            'prescriptions.patient',
            'prescriptions.doctor',
        ])->findOrFail($id);
    }

    public function render()
    {
        // Calculate summary statistics
        $totalStock = $this->drug->pharmacyStocks->sum('quantity');
        $availableStock = $this->drug->pharmacyStocks->filter(function($stock) {
            return !$stock->expiry_date || $stock->expiry_date->isFuture();
        })->sum('quantity');
        
        $expiredStock = $this->drug->pharmacyStocks->filter(function($stock) {
            return $stock->expiry_date && $stock->expiry_date->isPast();
        })->sum('quantity');

        $lowStockEntries = $this->drug->pharmacyStocks->filter(function($stock) {
            return $stock->isLowStock();
        });

        $expiringSoonEntries = $this->drug->pharmacyStocks->filter(function($stock) {
            return $stock->isExpiringSoon(30);
        });

        $prescriptionCount = $this->drug->prescriptions()->count();

        return view('livewire.pharmacy.drug-detail', [
            'totalStock' => $totalStock,
            'availableStock' => $availableStock,
            'expiredStock' => $expiredStock,
            'lowStockEntries' => $lowStockEntries,
            'expiringSoonEntries' => $expiringSoonEntries,
            'prescriptionCount' => $prescriptionCount,
        ]);
    }
}
