<?php

namespace App\Livewire\Patients;

use App\Models\Patient;
use App\Models\MedicalRecord;
use App\Services\AuditLogService;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('layouts.app')]
class PatientProfile extends Component
{
    public Patient $patient;
    public $editing = false;
    public $editingHistory = false;
    public $activeTab = 'overview'; // 'overview', 'history', 'appointments', 'prescriptions', 'audit'
    
    // Medical record form fields
    public $showRecordModal = false;
    public $editingRecord = null;
    public $record_visit_date = '';
    public $record_visit_type = 'consultation';
    public $record_symptoms = '';
    public $record_diagnosis = '';
    public $record_treatment_plan = '';
    public $record_notes = '';
    public $record_follow_up_date = '';

    // Form fields
    public $first_name;
    public $last_name;
    public $date_of_birth;
    public $gender;
    public $email;
    public $phone;
    public $address;
    public $emergency_contact_name;
    public $emergency_contact_phone;
    public $insurance_provider;
    public $insurance_number;
    public $medical_history;
    public $allergies;
    public $blood_type;
    public $height;
    public $weight;

    public function mount(Patient $patient)
    {
        $this->patient = $patient;
        $this->loadPatientData();
    }

    public function loadPatientData()
    {
        $this->first_name = $this->patient->first_name;
        $this->last_name = $this->patient->last_name;
        $this->date_of_birth = $this->patient->date_of_birth->format('Y-m-d');
        $this->gender = $this->patient->gender;
        $this->email = $this->patient->email;
        $this->phone = $this->patient->phone;
        $this->address = $this->patient->address;
        $this->emergency_contact_name = $this->patient->emergency_contact_name;
        $this->emergency_contact_phone = $this->patient->emergency_contact_phone;
        $this->insurance_provider = $this->patient->insurance_provider;
        $this->insurance_number = $this->patient->insurance_number;
        $this->medical_history = $this->patient->medical_history;
        $this->allergies = $this->patient->allergies;
        $this->blood_type = $this->patient->blood_type;
        $this->height = $this->patient->height;
        $this->weight = $this->patient->weight;
    }

    public function toggleEdit()
    {
        $this->editing = !$this->editing;
        if (!$this->editing) {
            $this->loadPatientData();
        }
    }

    public function save()
    {
        $this->validate([
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'date_of_birth' => ['required', 'date', 'before:today'],
            'gender' => ['required', 'in:male,female,other'],
            'email' => ['nullable', 'string', 'email', 'max:255'],
            'phone' => ['nullable', 'string', 'max:20'],
            'address' => ['nullable', 'string'],
            'emergency_contact_name' => ['nullable', 'string', 'max:255'],
            'emergency_contact_phone' => ['nullable', 'string', 'max:20'],
            'insurance_provider' => ['nullable', 'string', 'max:255'],
            'insurance_number' => ['nullable', 'string', 'max:255'],
            'medical_history' => ['nullable', 'string'],
            'allergies' => ['nullable', 'string'],
            'blood_type' => ['nullable', 'string', 'max:10'],
            'height' => ['nullable', 'numeric', 'min:50', 'max:250'],
            'weight' => ['nullable', 'numeric', 'min:10', 'max:500'],
        ]);

        $this->patient->update([
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'date_of_birth' => $this->date_of_birth,
            'gender' => $this->gender,
            'email' => $this->email,
            'phone' => $this->phone,
            'address' => $this->address,
            'emergency_contact_name' => $this->emergency_contact_name,
            'emergency_contact_phone' => $this->emergency_contact_phone,
            'insurance_provider' => $this->insurance_provider,
            'insurance_number' => $this->insurance_number,
            'medical_history' => $this->medical_history,
            'allergies' => $this->allergies,
            'blood_type' => $this->blood_type,
            'height' => $this->height,
            'weight' => $this->weight,
        ]);

        $this->editing = false;
        session()->flash('message', 'Patient profile updated successfully!');
    }

    public function toggleEditHistory()
    {
        $this->editingHistory = !$this->editingHistory;
        if (!$this->editingHistory) {
            $this->loadPatientData();
        }
    }

    public function saveHistory()
    {
        $this->validate([
            'medical_history' => ['nullable', 'string'],
            'allergies' => ['nullable', 'string'],
        ]);

        // Store old values for audit logging
        $oldMedicalHistory = $this->patient->medical_history;
        $oldAllergies = $this->patient->allergies;

        $this->patient->update([
            'medical_history' => $this->medical_history,
            'allergies' => $this->allergies,
        ]);

        // Log changes
        if ($oldMedicalHistory !== $this->medical_history) {
            AuditLogService::logMedicalHistoryUpdate(
                $this->patient,
                $oldMedicalHistory,
                $this->medical_history
            );
        }

        if ($oldAllergies !== $this->allergies) {
            AuditLogService::logAllergiesUpdate(
                $this->patient,
                $oldAllergies,
                $this->allergies
            );
        }

        $this->editingHistory = false;
        session()->flash('message', 'Medical history updated successfully!');
    }

    public function openRecordModal($recordId = null)
    {
        if ($recordId) {
            $record = MedicalRecord::findOrFail($recordId);
            $this->editingRecord = $record;
            $this->record_visit_date = $record->visit_date->format('Y-m-d');
            $this->record_visit_type = $record->visit_type;
            $this->record_symptoms = $record->symptoms;
            $this->record_diagnosis = $record->diagnosis;
            $this->record_treatment_plan = $record->treatment_plan;
            $this->record_notes = $record->notes;
            $this->record_follow_up_date = $record->follow_up_date ? $record->follow_up_date->format('Y-m-d') : '';
        } else {
            $this->resetRecordForm();
            $this->record_visit_date = now()->format('Y-m-d');
        }
        $this->showRecordModal = true;
    }

    public function closeRecordModal()
    {
        $this->showRecordModal = false;
        $this->editingRecord = null;
        $this->resetRecordForm();
    }

    public function resetRecordForm()
    {
        $this->record_visit_date = '';
        $this->record_visit_type = 'consultation';
        $this->record_symptoms = '';
        $this->record_diagnosis = '';
        $this->record_treatment_plan = '';
        $this->record_notes = '';
        $this->record_follow_up_date = '';
        $this->resetErrorBag();
    }

    public function saveRecord()
    {
        $this->validate([
            'record_visit_date' => ['required', 'date'],
            'record_visit_type' => ['required', 'in:consultation,follow_up,checkup,emergency'],
            'record_symptoms' => ['nullable', 'string'],
            'record_diagnosis' => ['nullable', 'string'],
            'record_treatment_plan' => ['nullable', 'string'],
            'record_notes' => ['nullable', 'string'],
            'record_follow_up_date' => ['nullable', 'date'],
        ]);

        $data = [
            'patient_id' => $this->patient->id,
            'doctor_id' => auth()->id(),
            'visit_date' => $this->record_visit_date,
            'visit_type' => $this->record_visit_type,
            'symptoms' => $this->record_symptoms ?: null,
            'diagnosis' => $this->record_diagnosis ?: null,
            'treatment_plan' => $this->record_treatment_plan ?: null,
            'notes' => $this->record_notes ?: null,
            'follow_up_date' => $this->record_follow_up_date ?: null,
        ];

        if ($this->editingRecord) {
            $this->editingRecord->update($data);
            session()->flash('message', 'Medical record updated successfully!');
        } else {
            MedicalRecord::create($data);
            session()->flash('message', 'Medical record created successfully!');
        }

        $this->closeRecordModal();
        $this->patient->refresh();
    }

    public function deleteRecord($recordId)
    {
        MedicalRecord::findOrFail($recordId)->delete();
        session()->flash('message', 'Medical record deleted successfully!');
        $this->patient->refresh();
    }

    public function setActiveTab($tab)
    {
        $this->activeTab = $tab;
        $this->editingHistory = false; // Reset editing state when switching tabs
    }

    public function render()
    {
        $this->patient->load([
            'medicalRecords.doctor', 
            'appointments.doctor', 
            'prescriptions.doctor',
            'prescriptions.drug',
            'consultations.doctor',
            'bills',
            'medications.prescriber'
        ]);
        
        return view('livewire.patients.patient-profile');
    }
}
