<?php

namespace App\Livewire\Nursing;

use App\Models\Admission;
use App\Models\MedicationAdministration;
use App\Models\Prescription;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class MedicationAdministrationLog extends Component
{
    use WithPagination;

    public $admissionId;
    public $admission;
    
    // Form fields
    public $showModal = false;
    public $prescription_id = '';
    public $medication_name = '';
    public $dosage = '';
    public $route = '';
    public $time_administered = '';
    public $notes = '';
    
    // Common medication routes
    public $availableRoutes = [
        'Oral',
        'IV',
        'IM',
        'Subcutaneous',
        'Topical',
        'Inhalation',
        'Rectal',
        'Sublingual',
        'Intranasal',
    ];

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Check if user is a nurse, developer, or cmd
        $user = auth()->user();
        if (!$user->isNurse() && !$user->isDeveloper() && !$user->isCmd()) {
            abort(403, 'Access denied. This page is for nursing staff, developers, and CMD only.');
        }

        // Check for admission ID in query string
        $admissionId = request()->query('admission');
        if ($admissionId) {
            $this->admissionId = $admissionId;
            $this->admission = Admission::with(['patient'])->findOrFail($admissionId);
        }
    }

    public function selectPrescription($prescriptionId)
    {
        $prescription = Prescription::with(['drug'])->findOrFail($prescriptionId);
        $this->prescription_id = $prescriptionId;
        if ($prescription->drug) {
            $this->medication_name = $prescription->drug->name;
        } else {
            $this->medication_name = '';
        }
        $this->dosage = $prescription->dosage;
        // Route is not stored in prescription, so it remains empty for manual entry
        $this->time_administered = now()->format('Y-m-d\TH:i');
        $this->showModal = true;
    }

    public function clearPrescription()
    {
        $this->prescription_id = '';
        $this->medication_name = '';
        $this->dosage = '';
        $this->route = '';
    }

    public function openModal()
    {
        $this->resetForm();
        $this->time_administered = now()->format('Y-m-d\TH:i');
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
        $this->resetErrorBag();
    }

    public function resetForm()
    {
        $this->prescription_id = '';
        $this->medication_name = '';
        $this->dosage = '';
        $this->route = '';
        $this->time_administered = '';
        $this->notes = '';
    }

    public function save()
    {
        $this->validate([
            'medication_name' => 'required|string|max:255',
            'dosage' => 'required|string|max:255',
            'route' => 'required|string|max:255',
            'time_administered' => 'required|date',
            'notes' => 'nullable|string|max:1000',
        ]);

        if (!$this->admission) {
            session()->flash('error', 'No admission selected.');
            return;
        }

        // Check if admission is still active
        if (!$this->admission->isActive()) {
            session()->flash('error', 'Cannot record medication administration for a discharged or transferred patient.');
            $this->closeModal();
            return;
        }

        MedicationAdministration::create([
            'admission_id' => $this->admissionId,
            'prescription_id' => $this->prescription_id ?: null,
            'medication_name' => $this->medication_name,
            'dosage' => $this->dosage,
            'route' => $this->route,
            'time_administered' => $this->time_administered,
            'nurse_id' => auth()->id(),
            'notes' => $this->notes ?: null,
        ]);

        session()->flash('message', 'Medication administration recorded successfully!');
        $this->closeModal();
        $this->resetPage();
    }

    public function delete($administrationId)
    {
        $administration = MedicationAdministration::findOrFail($administrationId);
        
        // Only allow deletion if it's the same nurse or user has admin privileges
        $user = auth()->user();
        if ($administration->nurse_id !== $user->id && !$user->isDeveloper() && !$user->isCmd()) {
            session()->flash('error', 'You can only delete your own medication administration records.');
            return;
        }

        $administration->delete();
        session()->flash('message', 'Medication administration record deleted successfully!');
        $this->resetPage();
    }

    public function render()
    {
        if (!$this->admission) {
            // If no admission ID provided, show list of active admissions
            $admissions = Admission::with(['patient'])
                ->where('status', 'admitted')
                ->orderBy('admission_date', 'desc')
                ->paginate(15);

            return view('livewire.nursing.medication-administration-log', [
                'admissions' => $admissions,
            ]);
        }

        // Load medication administrations for the selected admission
        $medicationAdministrations = MedicationAdministration::with(['nurse', 'prescription.drug'])
            ->where('admission_id', $this->admissionId)
            ->orderBy('time_administered', 'desc')
            ->paginate(15);

        // Load active prescriptions for this patient
        $prescriptions = Prescription::with(['drug', 'doctor'])
            ->where('patient_id', $this->admission->patient_id)
            ->active()
            ->orderBy('prescription_date', 'desc')
            ->get();

        return view('livewire.nursing.medication-administration-log', [
            'medicationAdministrations' => $medicationAdministrations,
            'admission' => $this->admission,
            'prescriptions' => $prescriptions,
        ]);
    }
}
