<?php

namespace App\Livewire\Labs;

use App\Models\LabRequest;
use App\Models\Visit;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class LabWorkflow extends Component
{
    use WithPagination;

    public $statusFilter = 'pending';
    public $selectedRequestId = null;
    public $selectedRequest = null;
    
    // Result recording fields
    public $showResultModal = false;
    public $results = '';
    public $result_data = [];
    public $notes = '';

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Check if user has lab access (lab staff, doctor, developer, or cmd)
        $user = auth()->user();
        if (!$user->isDoctor() && !$user->isLab() && !$user->isDeveloper() && !$user->isCmd()) {
            abort(403, 'Access denied. This page is for lab staff, doctors, developers, and CMD only.');
        }
    }

    /**
     * Check if user can modify lab requests (not read-only)
     * Doctors have read-only access
     */
    public function canModifyLabRequests(): bool
    {
        $user = auth()->user();
        return !$user->isDoctor() && ($user->isLab() || $user->isDeveloper() || $user->isCmd());
    }

    public function selectRequest($requestId)
    {
        $this->selectedRequestId = $requestId;
        $this->selectedRequest = LabRequest::with(['patient', 'visit', 'requestedBy'])->findOrFail($requestId);
        
        // Load existing results if any
        if ($this->selectedRequest->status === LabRequest::STATUS_COMPLETED) {
            $this->results = $this->selectedRequest->results ?? '';
            $this->result_data = $this->selectedRequest->result_data ?? [];
            $this->notes = $this->selectedRequest->notes ?? '';
        }
        
        $this->showResultModal = true;
    }

    public function closeResultModal()
    {
        $this->showResultModal = false;
        $this->resetResultForm();
    }

    public function resetResultForm()
    {
        $this->selectedRequestId = null;
        $this->selectedRequest = null;
        $this->results = '';
        $this->result_data = [];
        $this->notes = '';
        $this->resetErrorBag();
    }

    public function markInProgress($requestId)
    {
        if (!$this->canModifyLabRequests()) {
            abort(403, 'You do not have permission to modify lab requests.');
        }

        $request = LabRequest::findOrFail($requestId);
        $request->status = LabRequest::STATUS_IN_PROGRESS;
        $request->save();
        
        session()->flash('message', 'Lab request marked as in progress.');
        $this->resetPage();
    }

    public function saveResults()
    {
        if (!$this->canModifyLabRequests()) {
            abort(403, 'You do not have permission to modify lab requests.');
        }

        $this->validate([
            'results' => 'required|string|min:10',
        ]);

        if (!$this->selectedRequest) {
            session()->flash('error', 'No lab request selected.');
            return;
        }

        $this->selectedRequest->markCompleted(
            auth()->user(),
            $this->results,
            !empty($this->result_data) ? $this->result_data : null
        );

        if ($this->notes) {
            $this->selectedRequest->notes = $this->notes;
            $this->selectedRequest->save();
        }

        // Update visit status if all lab requests for this visit are completed
        $visit = $this->selectedRequest->visit;
        $pendingLabRequests = LabRequest::where('visit_id', $visit->id)
            ->where('status', '!=', LabRequest::STATUS_COMPLETED)
            ->where('status', '!=', LabRequest::STATUS_CANCELLED)
            ->count();

        if ($pendingLabRequests === 0 && $visit->status === Visit::STATUS_LAB_REQUESTED) {
            // All lab tests completed, can proceed
            $visit->updateStatus(Visit::STATUS_IN_CONSULTATION);
        }

        session()->flash('message', 'Lab results recorded successfully!');
        $this->closeResultModal();
        $this->resetPage();
    }

    public function cancelRequest($requestId)
    {
        if (!$this->canModifyLabRequests()) {
            abort(403, 'You do not have permission to modify lab requests.');
        }

        $request = LabRequest::findOrFail($requestId);
        $request->status = LabRequest::STATUS_CANCELLED;
        $request->save();
        
        session()->flash('message', 'Lab request cancelled.');
        $this->resetPage();
    }

    public function render()
    {
        $query = LabRequest::with(['patient', 'visit', 'requestedBy'])
            ->when($this->statusFilter === 'pending', function ($q) {
                $q->where('status', LabRequest::STATUS_PENDING);
            })
            ->when($this->statusFilter === 'in_progress', function ($q) {
                $q->where('status', LabRequest::STATUS_IN_PROGRESS);
            })
            ->when($this->statusFilter === 'completed', function ($q) {
                $q->where('status', LabRequest::STATUS_COMPLETED);
            })
            ->orderBy('requested_at', 'desc');

        $labRequests = $query->paginate(15);

        return view('livewire.labs.lab-workflow', [
            'labRequests' => $labRequests,
            'canModify' => $this->canModifyLabRequests(),
        ]);
    }
}
