<?php

namespace App\Livewire\Labs;

use App\Models\LabRequest;
use App\Models\Visit;
use Livewire\Component;

class LabRequestForm extends Component
{
    public $visitId;
    public $patientId;
    public $showModal = false;
    
    public $test_type = 'blood_test';
    public $test_name = '';
    public $clinical_indication = '';

    // Common test types
    public $commonTests = [
        'blood_test' => [
            'Complete Blood Count (CBC)',
            'Blood Glucose',
            'Lipid Profile',
            'Liver Function Test (LFT)',
            'Kidney Function Test (KFT)',
            'Thyroid Function Test (TFT)',
            'Hemoglobin A1C',
        ],
        'urine_test' => [
            'Urinalysis',
            'Urine Culture',
            'Pregnancy Test',
            'Drug Screening',
        ],
        'imaging' => [
            'Chest X-Ray',
            'Abdominal Ultrasound',
            'ECG',
            'CT Scan',
            'MRI',
        ],
        'other' => [
            'Stool Test',
            'Sputum Culture',
            'Throat Swab',
            'Other',
        ],
    ];

    public function mount($visitId = null, $patientId = null)
    {
        $this->visitId = $visitId;
        $this->patientId = $patientId;
        $this->showModal = true; // Auto-show modal when mounted
    }

    public function openModal()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->test_type = 'blood_test';
        $this->test_name = '';
        $this->clinical_indication = '';
        $this->resetErrorBag();
    }

    public function save()
    {
        $this->validate([
            'visitId' => 'required|exists:visits,id',
            'patientId' => 'required|exists:patients,id',
            'test_type' => 'required|string',
            'test_name' => 'required|string|max:255',
            'clinical_indication' => 'nullable|string|max:1000',
        ]);

        $labRequest = LabRequest::create([
            'visit_id' => $this->visitId,
            'patient_id' => $this->patientId,
            'requested_by' => auth()->id(),
            'test_type' => $this->test_type,
            'test_name' => $this->test_name,
            'clinical_indication' => $this->clinical_indication,
            'status' => LabRequest::STATUS_PENDING,
            'requested_at' => now(),
        ]);

        // Update visit status to lab_requested if not already
        $visit = Visit::findOrFail($this->visitId);
        if ($visit->status !== Visit::STATUS_LAB_REQUESTED) {
            $visit->updateStatus(Visit::STATUS_LAB_REQUESTED);
        }

        session()->flash('message', 'Lab request created successfully!');
        $this->closeModal();
        
        // Emit event to refresh parent component
        $this->dispatch('lab-request-created', ['visit_id' => $this->visitId]);
    }

    public function render()
    {
        return view('livewire.labs.lab-request-form');
    }
}
