<?php

namespace App\Livewire\Dashboard;

use App\Models\Patient;
use App\Models\Appointment;
use App\Models\Prescription;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Collection;

class RecentActivity extends Component
{
    use WithPagination;

    protected $paginationTheme = 'tailwind';

    public function render()
    {
        // Fetch recent activities from all sources
        $patients = Patient::latest()->limit(50)->get();
        $appointments = Appointment::with('patient')->latest()->limit(50)->get();
        $prescriptions = Prescription::with('patient')->latest()->limit(50)->get();
        
        $activities = collect();
        
        // Add patient registrations
        foreach ($patients as $patient) {
            $activities->push([
                'id' => 'patient-' . $patient->id,
                'type' => 'patient',
                'icon' => 'user',
                'color' => 'green',
                'message' => "New patient {$patient->full_name} registered",
                'time' => $patient->created_at,
                'link' => route('patients.show', $patient),
                'created_at' => $patient->created_at,
            ]);
        }
        
        // Add appointments
        foreach ($appointments as $appointment) {
            $activities->push([
                'id' => 'appointment-' . $appointment->id,
                'type' => 'appointment',
                'icon' => 'calendar',
                'color' => 'blue',
                'message' => "Appointment scheduled for {$appointment->patient_name}",
                'time' => $appointment->created_at,
                'link' => route('appointments.index'),
                'created_at' => $appointment->created_at,
            ]);
        }
        
        // Add prescriptions
        foreach ($prescriptions as $prescription) {
            $activities->push([
                'id' => 'prescription-' . $prescription->id,
                'type' => 'prescription',
                'icon' => 'pills',
                'color' => 'yellow',
                'message' => "Prescription issued for {$prescription->patient->full_name}",
                'time' => $prescription->created_at,
                'link' => route('prescriptions.index'),
                'created_at' => $prescription->created_at,
            ]);
        }
        
        // Sort by creation time (newest first)
        $sortedActivities = $activities->sortByDesc('created_at')->values();
        
        // Manual pagination for collection
        $perPage = 5;
        $currentPage = request()->get('page', 1);
        $total = $sortedActivities->count();
        $items = $sortedActivities->slice(($currentPage - 1) * $perPage, $perPage)->values();
        
        $paginatedActivities = new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $total,
            $perPage,
            $currentPage,
            ['path' => request()->url(), 'query' => request()->query()]
        );
        
        return view('livewire.dashboard.recent-activity', [
            'activities' => $paginatedActivities,
        ]);
    }
}
