<?php

namespace App\Livewire\Consultations;

use App\Models\Visit;
use App\Models\Consultation;
use App\Models\MedicalRecord;
use App\Models\LabRequest;
use App\Models\Admission;
use App\Models\Operation;
use App\Models\OperationRoom;
use App\Models\Prescription;
use App\Models\Drug;
use Illuminate\Support\Str;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class DoctorConsultation extends Component
{
    use WithPagination;

    public $selectedVisitId = null;
    public $selectedVisit = null;
    public $selectedPatient = null;
    public $vitalsRecord = null;
    
    // Consultation fields
    public $showConsultationModal = false;
    public $consultation_type = 'initial';
    public $chief_complaint = '';
    public $history_of_present_illness = '';
    public $physical_examination = '';
    public $assessment = '';
    public $plan = '';
    public $diagnosis = '';
    public $treatment_plan = '';
    public $follow_up_instructions = '';
    public $notes = '';
    public $duration = 30;
    
    // Lab request modal
    public $showLabRequestModal = false;
    
    // Admission modal
    public $showAdmissionModal = false;
    public $ward = '';
    public $room_number = '';
    public $bed_number = '';
    public $admission_reason = '';
    public $admission_notes = '';
    
    // Surgery booking modal
    public $showSurgeryModal = false;
    public $surgery_type = '';
    public $procedure_name = '';
    public $surgery_indication = '';
    public $operation_room_id = '';
    public $scheduled_date = '';
    public $scheduled_time = '';
    public $estimated_duration = 60;
    public $assistant_surgeon_id = '';
    public $anesthesiologist_id = '';
    public $surgery_notes = '';
    
    // Prescription modal
    public $showPrescriptionModal = false;
    public $prescriptionItems = [];
    public $prescription_date = '';
    public $prescription_valid_until = '';

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Check if user is a doctor, developer, or cmd
        $user = auth()->user();
        if (!$user->isDoctor() && !$user->isDeveloper() && !$user->isCmd()) {
            abort(403, 'Access denied. This page is for doctors, developers, and CMD only.');
        }
    }

    public function selectVisit($visitId)
    {
        $this->selectedVisitId = $visitId;
        $this->selectedVisit = Visit::with(['patient', 'medicalRecords'])->findOrFail($visitId);
        $this->selectedPatient = $this->selectedVisit->patient;
        
        // Load vitals from medical record for this visit
        $this->vitalsRecord = MedicalRecord::where('visit_id', $visitId)->first();
        
        // Load existing consultation if any
        $existingConsultation = Consultation::where('visit_id', $visitId)->first();
        if ($existingConsultation) {
            $this->chief_complaint = $existingConsultation->chief_complaint;
            $this->history_of_present_illness = $existingConsultation->history_of_present_illness ?? '';
            $this->physical_examination = $existingConsultation->physical_examination ?? '';
            $this->assessment = $existingConsultation->assessment ?? '';
            $this->plan = $existingConsultation->plan ?? '';
            $this->diagnosis = $existingConsultation->diagnosis ?? '';
            $this->treatment_plan = $existingConsultation->treatment_plan ?? '';
            $this->follow_up_instructions = $existingConsultation->follow_up_instructions ?? '';
            $this->notes = $existingConsultation->notes ?? '';
            $this->consultation_type = $existingConsultation->consultation_type;
            $this->duration = $existingConsultation->duration;
        } else {
            // Pre-fill from visit
            $this->chief_complaint = $this->selectedVisit->chief_complaint ?? '';
        }
        
        $this->showConsultationModal = true;
    }

    public function closeConsultationModal()
    {
        $this->showConsultationModal = false;
        $this->resetConsultationForm();
    }

    public function resetConsultationForm()
    {
        $this->selectedVisitId = null;
        $this->selectedVisit = null;
        $this->selectedPatient = null;
        $this->vitalsRecord = null;
        $this->consultation_type = 'initial';
        $this->chief_complaint = '';
        $this->history_of_present_illness = '';
        $this->physical_examination = '';
        $this->assessment = '';
        $this->plan = '';
        $this->diagnosis = '';
        $this->treatment_plan = '';
        $this->follow_up_instructions = '';
        $this->notes = '';
        $this->duration = 30;
        $this->resetErrorBag();
    }

    public function saveConsultation()
    {
        $this->validate([
            'chief_complaint' => 'required|string|max:1000',
            'consultation_type' => 'required|in:initial,follow_up,emergency,telemedicine,specialist',
            'duration' => 'required|integer|min:15|max:480',
        ]);

        if (!$this->selectedVisit) {
            session()->flash('error', 'No visit selected.');
            return;
        }

        // Get vitals from medical record
        $vitalSigns = $this->vitalsRecord && $this->vitalsRecord->vital_signs 
            ? $this->vitalsRecord->vital_signs 
            : null;

        $data = [
            'patient_id' => $this->selectedPatient->id,
            'visit_id' => $this->selectedVisitId,
            'doctor_id' => auth()->id(),
            'consultation_type' => $this->consultation_type,
            'chief_complaint' => $this->chief_complaint,
            'history_of_present_illness' => $this->history_of_present_illness ?: null,
            'physical_examination' => $this->physical_examination ?: null,
            'assessment' => $this->assessment ?: null,
            'plan' => $this->plan ?: null,
            'diagnosis' => $this->diagnosis ?: null,
            'treatment_plan' => $this->treatment_plan ?: null,
            'follow_up_instructions' => $this->follow_up_instructions ?: null,
            'consultation_date' => $this->selectedVisit->checked_in_at->toDateString(),
            'consultation_time' => $this->selectedVisit->checked_in_at,
            'duration' => $this->duration,
            'status' => 'completed',
            'notes' => $this->notes ?: null,
            'vital_signs' => $vitalSigns,
        ];

        $existingConsultation = Consultation::where('visit_id', $this->selectedVisitId)->first();
        
        if ($existingConsultation) {
            $existingConsultation->update($data);
            session()->flash('message', 'Consultation updated successfully!');
        } else {
            Consultation::create($data);
            session()->flash('message', 'Consultation created successfully!');
        }

        // Update visit status to in_consultation (or completed if no lab/prescription needed)
        $this->selectedVisit->updateStatus(Visit::STATUS_IN_CONSULTATION);

        $this->closeConsultationModal();
        $this->resetPage();
    }

    public function openLabRequestModal()
    {
        $this->showLabRequestModal = true;
    }

    public function closeLabRequestModal()
    {
        $this->showLabRequestModal = false;
    }

    public function openAdmissionModal()
    {
        if (!$this->selectedVisit) {
            session()->flash('error', 'Please select a visit first.');
            return;
        }
        
        // Pre-fill diagnosis from consultation if available
        $consultation = Consultation::where('visit_id', $this->selectedVisitId)->first();
        if ($consultation && $consultation->diagnosis) {
            $this->admission_reason = $consultation->diagnosis;
        } else {
            $this->admission_reason = $this->selectedVisit->chief_complaint ?? '';
        }
        
        $this->showAdmissionModal = true;
    }

    public function closeAdmissionModal()
    {
        $this->showAdmissionModal = false;
        $this->ward = '';
        $this->room_number = '';
        $this->bed_number = '';
        $this->admission_reason = '';
        $this->admission_notes = '';
        $this->resetErrorBag();
    }

    public function saveAdmission()
    {
        $this->validate([
            'ward' => 'required|string|max:255',
            'room_number' => 'nullable|string|max:50',
            'bed_number' => 'nullable|string|max:50',
            'admission_reason' => 'required|string|min:10|max:2000',
            'admission_notes' => 'nullable|string|max:2000',
        ], [
            'admission_reason.required' => 'Please provide a reason for admission.',
            'admission_reason.min' => 'Admission reason must be at least 10 characters.',
        ]);

        if (!$this->selectedVisit || !$this->selectedPatient) {
            session()->flash('error', 'No visit or patient selected.');
            return;
        }

        // Check if patient already has an active admission
        $activeAdmission = $this->selectedPatient->active_admission;
        if ($activeAdmission) {
            session()->flash('error', 'Patient already has an active admission: ' . $activeAdmission->admission_number);
            $this->closeAdmissionModal();
            return;
        }

        // Get diagnosis from consultation if available
        $consultation = Consultation::where('visit_id', $this->selectedVisitId)->first();
        $diagnosis = $consultation ? $consultation->diagnosis : null;

        // Create admission
        $admission = Admission::create([
            'patient_id' => $this->selectedPatient->id,
            'visit_id' => $this->selectedVisitId,
            'admitted_by' => auth()->id(),
            'ward' => $this->ward,
            'room_number' => $this->room_number ?: null,
            'bed_number' => $this->bed_number ?: null,
            'admission_reason' => $this->admission_reason,
            'diagnosis' => $diagnosis,
            'admission_notes' => $this->admission_notes ?: null,
            'status' => 'admitted',
            'admission_date' => now(),
        ]);

        // Update visit status to admitted
        $this->selectedVisit->updateStatus(Visit::STATUS_ADMITTED);

        session()->flash('message', 'Patient admitted successfully! Admission Number: ' . $admission->admission_number);
        $this->closeAdmissionModal();
        $this->resetPage();
    }

    public function openSurgeryModal()
    {
        if (!$this->selectedVisit) {
            session()->flash('error', 'Please select a visit first.');
            return;
        }
        
        // Pre-fill diagnosis and indication from consultation if available
        $consultation = Consultation::where('visit_id', $this->selectedVisitId)->first();
        if ($consultation) {
            $this->surgery_indication = $consultation->diagnosis ?? $this->selectedVisit->chief_complaint ?? '';
            $this->diagnosis = $consultation->diagnosis ?? '';
        } else {
            $this->surgery_indication = $this->selectedVisit->chief_complaint ?? '';
        }
        
        $this->scheduled_date = now()->addDays(1)->format('Y-m-d');
        $this->scheduled_time = '09:00';
        $this->showSurgeryModal = true;
    }

    public function closeSurgeryModal()
    {
        $this->showSurgeryModal = false;
        $this->surgery_type = '';
        $this->procedure_name = '';
        $this->surgery_indication = '';
        $this->operation_room_id = '';
        $this->scheduled_date = '';
        $this->scheduled_time = '';
        $this->estimated_duration = 60;
        $this->assistant_surgeon_id = '';
        $this->anesthesiologist_id = '';
        $this->surgery_notes = '';
        $this->resetErrorBag();
    }

    public function bookSurgery()
    {
        $this->validate([
            'surgery_type' => 'required|string|max:255',
            'procedure_name' => 'required|string|max:500',
            'surgery_indication' => 'required|string|min:10|max:2000',
            'scheduled_date' => 'required|date|after_or_equal:today',
            'scheduled_time' => 'required',
            'estimated_duration' => 'required|integer|min:15|max:1440',
            'operation_room_id' => 'nullable|exists:operation_rooms,id',
        ], [
            'surgery_indication.required' => 'Please provide an indication for surgery.',
            'surgery_indication.min' => 'Surgery indication must be at least 10 characters.',
        ]);

        if (!$this->selectedVisit || !$this->selectedPatient) {
            session()->flash('error', 'No visit or patient selected.');
            return;
        }

        // Check room availability if room is selected
        if ($this->operation_room_id) {
            $room = OperationRoom::find($this->operation_room_id);
            if ($room && !$room->isAvailableFor($this->scheduled_date, $this->scheduled_time, $this->estimated_duration)) {
                session()->flash('error', 'Operation room is not available at the selected date and time.');
                return;
            }
        }

        // Get diagnosis from consultation if available
        $consultation = Consultation::where('visit_id', $this->selectedVisitId)->first();
        $diagnosis = $consultation ? $consultation->diagnosis : null;

        // Create surgery booking
        $operation = Operation::create([
            'patient_id' => $this->selectedPatient->id,
            'visit_id' => $this->selectedVisitId,
            'surgeon_id' => auth()->id(),
            'assistant_surgeon_id' => $this->assistant_surgeon_id ?: null,
            'anesthesiologist_id' => $this->anesthesiologist_id ?: null,
            'operation_room_id' => $this->operation_room_id ?: null,
            'surgery_type' => ucwords(strtolower($this->surgery_type)),
            'procedure_name' => ucwords(strtolower($this->procedure_name)),
            'diagnosis' => $diagnosis,
            'indication' => $this->surgery_indication,
            'scheduled_date' => $this->scheduled_date,
            'scheduled_time' => $this->scheduled_time,
            'estimated_duration' => $this->estimated_duration,
            'status' => Operation::STATUS_SCHEDULED,
            'notes' => $this->surgery_notes ?: null,
            'booked_by' => auth()->id(),
        ]);

        // Update operation room status if room is assigned
        if ($this->operation_room_id) {
            OperationRoom::find($this->operation_room_id)->update(['status' => OperationRoom::STATUS_OCCUPIED]);
        }

        session()->flash('message', 'Surgery booked successfully! Operation Number: ' . $operation->operation_number);
        $this->closeSurgeryModal();
        $this->resetPage();
    }

    public function openPrescriptionModal()
    {
        if (!$this->selectedVisit || !$this->selectedPatient) {
            session()->flash('error', 'Please select a visit first.');
            return;
        }
        
        $this->resetPrescriptionForm();
        $this->prescription_date = now()->format('Y-m-d');
        $this->prescription_valid_until = now()->addDays(30)->format('Y-m-d');
        $this->addPrescriptionRow();
        $this->showPrescriptionModal = true;
    }

    public function closePrescriptionModal()
    {
        $this->showPrescriptionModal = false;
        $this->resetPrescriptionForm();
        $this->resetErrorBag();
    }

    public function resetPrescriptionForm()
    {
        $this->prescriptionItems = [];
        $this->prescription_date = '';
        $this->prescription_valid_until = '';
    }

    public function addPrescriptionRow()
    {
        $this->prescriptionItems[] = [
            'drug_id' => '',
            'dosage' => '',
            'frequency' => '',
            'duration' => 7,
            'quantity' => '',
            'instructions' => '',
            'status' => 'active',
            'notes' => '',
            'max_refills' => 0,
        ];
    }

    public function removePrescriptionRow($index)
    {
        if (isset($this->prescriptionItems[$index])) {
            unset($this->prescriptionItems[$index]);
            $this->prescriptionItems = array_values($this->prescriptionItems);
        }
    }

    public function savePrescription()
    {
        // Validate prescriptions array
        $this->validate([
            'prescriptionItems' => 'required|array|min:1',
            'prescriptionItems.*.drug_id' => 'required|exists:drugs,id',
            'prescriptionItems.*.dosage' => 'required|string|max:255',
            'prescriptionItems.*.frequency' => 'required|string|max:255',
            'prescriptionItems.*.duration' => 'required|integer|min:1|max:365',
            'prescriptionItems.*.quantity' => 'required|integer|min:1',
            'prescription_date' => 'required|date',
            'prescription_valid_until' => 'required|date|after:prescription_date',
            'prescriptionItems.*.status' => 'required|in:active,dispensed,expired,cancelled',
            'prescriptionItems.*.max_refills' => 'required|integer|min:0|max:12',
        ], [], [
            'prescriptionItems' => 'prescription list',
            'prescriptionItems.*.drug_id' => 'drug',
            'prescriptionItems.*.dosage' => 'dosage',
            'prescriptionItems.*.frequency' => 'frequency',
            'prescriptionItems.*.duration' => 'duration',
            'prescriptionItems.*.quantity' => 'quantity',
            'prescription_date' => 'prescription date',
            'prescription_valid_until' => 'valid until',
            'prescriptionItems.*.status' => 'status',
            'prescriptionItems.*.max_refills' => 'max refills',
        ]);

        if (!$this->selectedVisit || !$this->selectedPatient) {
            session()->flash('error', 'No visit or patient selected.');
            return;
        }

        // Filter out empty prescriptions
        $validPrescriptions = array_filter($this->prescriptionItems, function ($prescription) {
            return !empty($prescription['drug_id']);
        });

        if (empty($validPrescriptions)) {
            session()->flash('error', 'Please add at least one prescription.');
            return;
        }

        $prescriptionNumbers = [];
        
        // Create all prescriptions
        foreach ($validPrescriptions as $prescriptionData) {
            $prescription = Prescription::create([
                'patient_id' => $this->selectedPatient->id,
                'visit_id' => $this->selectedVisitId,
                'doctor_id' => auth()->id(),
                'drug_id' => $prescriptionData['drug_id'],
                'prescription_number' => 'RX-' . strtoupper(Str::random(8)),
                'dosage' => $prescriptionData['dosage'],
                'frequency' => $prescriptionData['frequency'],
                'duration' => $prescriptionData['duration'],
                'quantity' => $prescriptionData['quantity'],
                'instructions' => $prescriptionData['instructions'] ?? null,
                'prescription_date' => $this->prescription_date,
                'valid_until' => $this->prescription_valid_until,
                'status' => $prescriptionData['status'],
                'notes' => $prescriptionData['notes'] ?? null,
                'refill_count' => 0,
                'max_refills' => $prescriptionData['max_refills'],
            ]);
            
            $prescriptionNumbers[] = $prescription->prescription_number;
        }

        // Update visit status if needed
        if ($this->selectedVisit->status !== Visit::STATUS_PRESCRIPTION_ISSUED) {
            $this->selectedVisit->updateStatus(Visit::STATUS_PRESCRIPTION_ISSUED);
        }

        $count = count($validPrescriptions);
        $message = $count === 1 
            ? 'Prescription created successfully! Prescription Number: ' . $prescriptionNumbers[0]
            : $count . ' prescriptions created successfully!';
        
        session()->flash('message', $message);
        $this->closePrescriptionModal();
        // Don't reset page or close consultation modal - keep consultation modal open
    }

    public function getLabRequestsProperty()
    {
        if (!$this->selectedVisitId) {
            return collect();
        }
        return LabRequest::where('visit_id', $this->selectedVisitId)
            ->with(['requestedBy', 'completedBy'])
            ->get();
    }

    public function getPrescriptionsProperty()
    {
        if (!$this->selectedVisitId) {
            return collect();
        }
        return Prescription::where('visit_id', $this->selectedVisitId)
            ->with(['drug', 'doctor'])
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function getPastConsultationsProperty()
    {
        if (!$this->selectedPatient) {
            return collect();
        }
        // Get all consultations for this patient, excluding the current visit's consultation
        $query = Consultation::where('patient_id', $this->selectedPatient->id);
        
        // Exclude current visit's consultation if we have a selected visit
        if ($this->selectedVisitId) {
            $query->where('visit_id', '!=', $this->selectedVisitId);
        }
        
        return $query->with(['doctor', 'visit'])
            ->orderBy('consultation_date', 'desc')
            ->orderBy('consultation_time', 'desc')
            ->limit(10)
            ->get();
    }

    public function render()
    {
        // Get visits waiting for doctor
        $visits = Visit::with(['patient', 'createdBy', 'medicalRecords'])
            ->whereIn('status', [Visit::STATUS_WAITING_DOCTOR, Visit::STATUS_IN_CONSULTATION])
            ->whereNull('checked_out_at')
            ->orderBy('checked_in_at', 'asc')
            ->paginate(15);

        // Load drugs for prescription modal
        $drugs = Drug::where('is_active', true)->orderBy('name')->get();

        return view('livewire.consultations.doctor-consultation', [
            'visits' => $visits,
            'drugs' => $drugs,
        ]);
    }
}
