<?php

namespace App\Livewire\Billing;

use App\Models\Bill;
use App\Models\Payment;
use App\Models\Patient;
use App\Models\User;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;
use Illuminate\Support\Str;

#[Layout('layouts.app')]
class BillList extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = '';
    public $showCreateModal = false;
    public $showPaymentModal = false;
    public $selectedBill = null;

    // Bill form fields
    public $patient_id = '';
    public $subtotal = 0;
    public $tax_amount = 0;
    public $discount_amount = 0;
    public $total_amount = 0;
    public $due_date = '';
    public $status = 'pending';
    public $notes = '';
    public $insurance_covered = false;
    public $patient_responsibility = 0;
    
    // Patient search
    public $patientSearch = '';
    public $searchResults = [];

    // Payment form fields
    public $payment_amount = 0;
    public $payment_method = 'cash';
    public $payment_reference = '';
    public $payment_notes = '';

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => ''],
    ];

    public function updatedSubtotal()
    {
        $this->calculateTotal();
    }

    public function updatedTaxAmount()
    {
        $this->calculateTotal();
    }

    public function updatedDiscountAmount()
    {
        $this->calculateTotal();
    }

    public function calculateTotal()
    {
        $this->total_amount = $this->subtotal + $this->tax_amount - $this->discount_amount;
        if ($this->insurance_covered) {
            $this->patient_responsibility = $this->total_amount * 0.2; // Assuming 20% patient responsibility
        } else {
            $this->patient_responsibility = $this->total_amount;
        }
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->due_date = now()->addDays(30)->format('Y-m-d');
        $this->showCreateModal = true;
    }

    public function openPaymentModal($billId)
    {
        $this->selectedBill = Bill::with('payments')->findOrFail($billId);
        $this->payment_amount = $this->selectedBill->balance;
        $this->showPaymentModal = true;
    }

    public function closeModals()
    {
        $this->showCreateModal = false;
        $this->showPaymentModal = false;
        $this->selectedBill = null;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->patient_id = '';
        $this->subtotal = 0;
        $this->tax_amount = 0;
        $this->discount_amount = 0;
        $this->total_amount = 0;
        $this->due_date = '';
        $this->status = 'pending';
        $this->notes = '';
        $this->insurance_covered = false;
        $this->patient_responsibility = 0;
        $this->payment_amount = 0;
        $this->payment_method = 'cash';
        $this->payment_reference = '';
        $this->payment_notes = '';
        $this->patientSearch = '';
        $this->searchResults = [];
    }
    
    public function searchPatients()
    {
        if (strlen($this->patientSearch) < 2) {
            $this->searchResults = [];
            return;
        }

        $searchTerm = strtolower($this->patientSearch);
        $this->searchResults = Patient::query()
            ->where(function ($query) use ($searchTerm) {
                $query->whereRaw('LOWER(first_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(last_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(patient_id) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(phone) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(email) like ?', ['%' . $searchTerm . '%']);
            })
            ->limit(10)
            ->get();
    }

    public function selectPatient($patientId)
    {
        $this->patient_id = $patientId;
        $patient = Patient::find($patientId);
        if ($patient) {
            $this->patientSearch = $patient->full_name;
        }
        $this->searchResults = [];
    }

    public function saveBill()
    {
        $this->validate([
            'patient_id' => 'required|exists:patients,id',
            'subtotal' => 'required|numeric|min:0',
            'tax_amount' => 'required|numeric|min:0',
            'discount_amount' => 'required|numeric|min:0',
            'total_amount' => 'required|numeric|min:0',
            'due_date' => 'required|date',
            'status' => 'required|in:pending,paid,partial,overdue,cancelled',
            'patient_responsibility' => 'required|numeric|min:0',
        ]);

        $this->calculateTotal();

        Bill::create([
            'patient_id' => $this->patient_id,
            'bill_number' => 'BILL-' . strtoupper(Str::random(8)),
            'subtotal' => $this->subtotal,
            'tax_amount' => $this->tax_amount,
            'discount_amount' => $this->discount_amount,
            'total_amount' => $this->total_amount,
            'due_date' => $this->due_date,
            'status' => $this->status,
            'notes' => $this->notes,
            'created_by' => auth()->id(),
            'billing_period_start' => now()->startOfMonth(),
            'billing_period_end' => now()->endOfMonth(),
            'insurance_covered' => $this->insurance_covered,
            'patient_responsibility' => $this->patient_responsibility,
        ]);

        session()->flash('message', 'Bill created successfully!');
        $this->closeModals();
    }

    public function savePayment()
    {
        $this->validate([
            'payment_amount' => 'required|numeric|min:0.01|max:' . $this->selectedBill->balance,
            'payment_method' => 'required|in:cash,card,bank_transfer,insurance,cheque',
            'payment_reference' => 'nullable|string|max:255',
        ]);

        $payment = Payment::create([
            'bill_id' => $this->selectedBill->id,
            'patient_id' => $this->selectedBill->patient_id,
            'amount' => $this->payment_amount,
            'payment_method' => $this->payment_method,
            'payment_reference' => $this->payment_reference,
            'payment_date' => now(),
            'status' => 'completed',
            'notes' => $this->payment_notes,
            'processed_by' => auth()->id(),
            'net_amount' => $this->payment_amount,
        ]);

        // Update bill status
        $bill = $this->selectedBill;
        $paidAmount = $bill->payments()->where('status', 'completed')->sum('amount');
        if ($paidAmount >= $bill->total_amount) {
            $bill->update(['status' => 'paid']);
        } elseif ($paidAmount > 0) {
            $bill->update(['status' => 'partial']);
        }

        session()->flash('message', 'Payment processed successfully!');
        $this->closeModals();
    }

    public function render()
    {
        $bills = Bill::query()
            ->with(['patient', 'payments'])
            ->when($this->search, function ($query) {
                $query->whereHas('patient', function ($q) {
                    $q->where('first_name', 'like', '%' . $this->search . '%')
                      ->orWhere('last_name', 'like', '%' . $this->search . '%')
                      ->orWhere('patient_id', 'like', '%' . $this->search . '%');
                })->orWhere('bill_number', 'like', '%' . $this->search . '%');
            })
            ->when($this->statusFilter, function ($query) {
                $query->where('status', $this->statusFilter);
            })
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        $patients = Patient::orderBy('first_name')->get();

        return view('livewire.billing.bill-list', [
            'bills' => $bills,
            'patients' => $patients,
        ]);
    }
}

