<?php

namespace App\Livewire\Appointments;

use App\Models\Appointment;
use App\Models\Patient;
use App\Models\User;
use App\Models\Visit;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

#[Layout('layouts.app')]
class AppointmentList extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = '';
    public $typeFilter = '';
    public $dateFilter = '';
    public $sortBy = 'appointment_date';
    public $sortDirection = 'desc';
    public $showCreateModal = false;
    public $showEditModal = false;
    public $editingAppointment = null;
    public $viewMode = 'list'; // 'list' or 'calendar'
    public $calendarMonth;
    public $calendarYear;

    // Form fields
    public $is_guest = false;
    public $patient_id = '';
    public $guest_name = '';
    public $guest_phone = '';
    public $guest_email = '';
    public $doctor_id = '';
    public $appointment_date = '';
    public $appointment_time = '';
    public $duration = 30;
    public $type = 'consultation';
    public $status = 'scheduled';
    public $notes = '';
    public $reason = '';
    
    // Patient search for non-guest appointments
    public $patientSearch = '';
    public $searchResults = [];

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'typeFilter' => ['except' => ''],
        'dateFilter' => ['except' => ''],
        'viewMode' => ['except' => 'list'],
        'calendarMonth',
        'calendarYear',
    ];

    public function mount()
    {
        $this->calendarMonth = now()->month;
        $this->calendarYear = now()->year;
    }

    public function switchView($mode)
    {
        $this->viewMode = $mode;
        $this->resetPage();
    }

    public function previousMonth()
    {
        $date = Carbon::create($this->calendarYear, $this->calendarMonth, 1)->subMonth();
        $this->calendarMonth = $date->month;
        $this->calendarYear = $date->year;
    }

    public function nextMonth()
    {
        $date = Carbon::create($this->calendarYear, $this->calendarMonth, 1)->addMonth();
        $this->calendarMonth = $date->month;
        $this->calendarYear = $date->year;
    }

    public function goToToday()
    {
        $this->calendarMonth = now()->month;
        $this->calendarYear = now()->year;
    }

    protected function rules()
    {
        $rules = [
            'doctor_id' => 'required|exists:users,id',
            'appointment_date' => 'required|date',
            'appointment_time' => 'required',
            'duration' => 'required|integer|min:15|max:480',
            'type' => 'required|in:consultation,follow_up,checkup,emergency',
            'status' => 'required|in:scheduled,confirmed,in_progress,completed,cancelled,no_show',
            'notes' => 'nullable|string',
            'reason' => 'nullable|string',
        ];

        if ($this->is_guest) {
            $rules['guest_name'] = 'required|string|max:255';
            $rules['guest_phone'] = 'required|string|max:20';
            $rules['guest_email'] = 'nullable|email|max:255';
        } else {
            $rules['patient_id'] = 'required|exists:patients,id';
        }

        return $rules;
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->showCreateModal = true;
    }

    public function openEditModal($appointmentId)
    {
        $appointment = Appointment::findOrFail($appointmentId);
        $this->editingAppointment = $appointment;
        $this->is_guest = $appointment->isGuest();
        $this->patient_id = $appointment->patient_id ?? '';
        if ($this->patient_id && !$this->is_guest) {
            $patient = Patient::find($this->patient_id);
            if ($patient) {
                $this->patientSearch = $patient->full_name . ' (' . $patient->patient_id . ')';
            }
        }
        $this->guest_name = $appointment->guest_name ?? '';
        $this->guest_phone = $appointment->guest_phone ?? '';
        $this->guest_email = $appointment->guest_email ?? '';
        $this->doctor_id = $appointment->doctor_id;
        $this->appointment_date = $appointment->appointment_date->format('Y-m-d');
        $this->appointment_time = $appointment->appointment_time;
        $this->duration = $appointment->duration;
        $this->type = $appointment->type;
        $this->status = $appointment->status;
        $this->notes = $appointment->notes;
        $this->reason = $appointment->reason;
        $this->showEditModal = true;
    }

    public function closeModals()
    {
        $this->showCreateModal = false;
        $this->showEditModal = false;
        $this->editingAppointment = null;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->is_guest = false;
        $this->patient_id = '';
        $this->guest_name = '';
        $this->guest_phone = '';
        $this->guest_email = '';
        $this->doctor_id = '';
        $this->appointment_date = '';
        $this->appointment_time = '';
        $this->duration = 30;
        $this->type = 'consultation';
        $this->status = 'scheduled';
        $this->notes = '';
        $this->reason = '';
        $this->patientSearch = '';
        $this->searchResults = [];
    }
    
    public function searchPatients()
    {
        if (strlen($this->patientSearch) < 2) {
            $this->searchResults = [];
            return;
        }

        $searchTerm = strtolower($this->patientSearch);
        $this->searchResults = Patient::query()
            ->where(function ($query) use ($searchTerm) {
                $query->whereRaw('LOWER(first_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(last_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(patient_id) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(phone) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(email) like ?', ['%' . $searchTerm . '%']);
            })
            ->limit(10)
            ->get();
    }

    public function selectPatient($patientId)
    {
        $this->patient_id = $patientId;
        $patient = Patient::find($patientId);
        if ($patient) {
            $this->patientSearch = $patient->full_name . ' (' . $patient->patient_id . ')';
        }
        $this->searchResults = [];
    }

    public function save()
    {
        $this->validate();

        $appointmentTime = $this->appointment_time;

        $data = [
            'doctor_id' => $this->doctor_id,
            'appointment_date' => $this->appointment_date,
            'appointment_time' => $appointmentTime,
            'duration' => $this->duration,
            'type' => $this->type,
            'status' => $this->status,
            'notes' => $this->notes,
            'reason' => $this->reason,
        ];

        if ($this->is_guest) {
            $data['patient_id'] = null;
            $data['guest_name'] = $this->guest_name;
            $data['guest_phone'] = $this->guest_phone;
            $data['guest_email'] = $this->guest_email ?: null;
        } else {
            $data['patient_id'] = $this->patient_id;
            $data['guest_name'] = null;
            $data['guest_phone'] = null;
            $data['guest_email'] = null;
        }

        if ($this->editingAppointment) {
            $this->editingAppointment->update($data);
            session()->flash('message', 'Appointment updated successfully!');
        } else {
            Appointment::create($data);
            session()->flash('message', 'Appointment created successfully!');
        }

        $this->closeModals();
    }

    public function delete($appointmentId)
    {
        Appointment::findOrFail($appointmentId)->delete();
        session()->flash('message', 'Appointment deleted successfully!');
    }

    public function checkIn($appointmentId)
    {
        $appointment = Appointment::with('doctor')->findOrFail($appointmentId);
        
        if ($appointment->status === Appointment::STATUS_COMPLETED || $appointment->status === Appointment::STATUS_CANCELLED) {
            session()->flash('error', 'Cannot check in a ' . $appointment->status . ' appointment.');
            return;
        }

        try {
            DB::beginTransaction();

            $patient = null;

            // If this is a guest appointment, create a new patient first
            if ($appointment->isGuest()) {
                // Generate patient ID (format: PXXXXXX where X is random)
                $patientId = 'P' . strtoupper(Str::random(6));
                
                // Split guest name into first and last name
                $nameParts = explode(' ', trim($appointment->guest_name), 2);
                $firstName = $nameParts[0] ?? $appointment->guest_name;
                $lastName = $nameParts[1] ?? '';
                
                // Create new patient from guest information
                $patient = Patient::create([
                    'user_id' => auth()->id(), // Track which staff member created this patient
                    'patient_id' => $patientId,
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                    'date_of_birth' => now()->subYears(30)->format('Y-m-d'), // Default age if not provided
                    'gender' => 'other', // Default gender
                    'phone' => $appointment->guest_phone,
                    'email' => $appointment->guest_email,
                    'address' => null,
                ]);

                // Update appointment to link to the new patient
                $appointment->update([
                    'patient_id' => $patient->id,
                    'guest_name' => null,
                    'guest_phone' => null,
                    'guest_email' => null,
                ]);
            } else {
                // Get the existing patient
                $patient = $appointment->patient;
                if (!$patient) {
                    throw new \Exception('Patient not found for this appointment.');
                }
            }

            // Create visit from appointment (visit number will be auto-generated)
            $visit = Visit::create([
                'patient_id' => $patient->id,
                'created_by' => auth()->id(),
                'status' => Visit::STATUS_CHECKED_IN,
                'chief_complaint' => $appointment->reason ?? 'Appointment check-in',
                'visit_notes' => "Checked in from appointment scheduled for {$appointment->appointment_date->format('M d, Y')} at {$appointment->appointment_time}",
            ]);

            // Update appointment status
            $appointment->update([
                'status' => Appointment::STATUS_CONFIRMED,
                'checked_in_at' => now(),
            ]);

            DB::commit();

            session()->flash('message', 'Appointment checked in successfully! Visit #' . $visit->visit_number . ' created.');
        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('error', 'Error checking in appointment: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $baseQuery = Appointment::query()
            ->with(['patient', 'doctor'])
            ->when($this->search, function ($query) {
                $query->whereHas('patient', function ($q) {
                    $q->where('first_name', 'like', '%' . $this->search . '%')
                      ->orWhere('last_name', 'like', '%' . $this->search . '%')
                      ->orWhere('patient_id', 'like', '%' . $this->search . '%');
                })->orWhereHas('doctor', function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->statusFilter, function ($query) {
                $query->where('status', $this->statusFilter);
            })
            ->when($this->typeFilter, function ($query) {
                $query->where('type', $this->typeFilter);
            })
            ->when($this->dateFilter, function ($query) {
                $query->whereDate('appointment_date', $this->dateFilter);
            });

        if ($this->viewMode === 'calendar') {
            // For calendar view, get all appointments for the month
            $startOfMonth = Carbon::create($this->calendarYear, $this->calendarMonth, 1)->startOfMonth();
            $endOfMonth = $startOfMonth->copy()->endOfMonth();
            
            $appointments = $baseQuery->whereBetween('appointment_date', [$startOfMonth, $endOfMonth])
                ->orderBy('appointment_date')
                ->orderBy('appointment_time')
                ->get();
            
            // Group appointments by date
            $appointmentsByDate = $appointments->groupBy(function($appointment) {
                return $appointment->appointment_date->format('Y-m-d');
            });
            
            // Build calendar grid
            $calendarDays = [];
            $daysInMonth = $startOfMonth->daysInMonth;
            $firstDayOfWeek = $startOfMonth->dayOfWeek; // 0 (Sunday) to 6 (Saturday)
            
            // Add empty cells for days before month starts
            for ($i = 0; $i < $firstDayOfWeek; $i++) {
                $calendarDays[] = null;
            }
            
            // Add days of the month
            for ($day = 1; $day <= $daysInMonth; $day++) {
                $date = Carbon::create($this->calendarYear, $this->calendarMonth, $day);
                $dateKey = $date->format('Y-m-d');
                $calendarDays[] = [
                    'date' => $date,
                    'day' => $day,
                    'appointments' => $appointmentsByDate->get($dateKey, collect())
                ];
            }
        } else {
            // For list view, use pagination
            $appointments = $baseQuery->orderBy($this->sortBy, $this->sortDirection)->paginate(15);
            $appointmentsByDate = collect();
            $calendarDays = [];
        }

        $patients = Patient::orderBy('first_name')->get();
        // Only show doctors (users with role 'doctor', 'staff', or null for backward compatibility)
        $doctors = User::where(function($query) {
            $query->where('role', 'doctor')
                  ->orWhere('role', 'staff')
                  ->orWhereNull('role');
        })->orderBy('name')->get();

        return view('livewire.appointments.appointment-list', [
            'appointments' => $appointments,
            'patients' => $patients,
            'doctors' => $doctors,
            'calendarDays' => $calendarDays,
            'calendarMonthName' => Carbon::create($this->calendarYear, $this->calendarMonth, 1)->format('F Y'),
        ]);
    }
}

