<?php

namespace App\Livewire\AI;

use App\Services\AIDiagnosisService;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('layouts.app')]
class AIDiagnosisAssistant extends Component
{
    public $symptoms = [];
    public $symptomInput = '';
    public $vitalSigns = [
        'blood_pressure' => '',
        'temperature' => '',
        'heart_rate' => '',
        'respiratory_rate' => '',
        'oxygen_saturation' => '',
    ];
    public $patientHistory = [];
    public $historyInput = '';
    public $diagnosisResults = null;
    public $isLoading = false;

    protected $rules = [
        'symptoms' => 'required|array|min:1',
    ];

    protected $messages = [
        'symptoms.required' => 'Please enter at least one symptom.',
        'symptoms.min' => 'Please enter at least one symptom.',
    ];

    public function addSymptom()
    {
        if (!empty($this->symptomInput)) {
            $this->symptoms[] = trim($this->symptomInput);
            $this->symptomInput = '';
        }
    }

    public function removeSymptom($index)
    {
        unset($this->symptoms[$index]);
        $this->symptoms = array_values($this->symptoms);
    }

    public function addHistory()
    {
        if (!empty($this->historyInput)) {
            $this->patientHistory[] = trim($this->historyInput);
            $this->historyInput = '';
        }
    }

    public function removeHistory($index)
    {
        unset($this->patientHistory[$index]);
        $this->patientHistory = array_values($this->patientHistory);
    }

    public function getDiagnosisSuggestions()
    {
        $this->validate();
        
        $this->isLoading = true;
        $this->diagnosisResults = null;

        try {
            $aiService = new AIDiagnosisService();
            $this->diagnosisResults = $aiService->getDiagnosisSuggestions(
                $this->symptoms,
                array_filter($this->vitalSigns),
                $this->patientHistory
            );
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to get diagnosis suggestions. Please try again.');
        } finally {
            $this->isLoading = false;
        }
    }

    public function getUrgencyLevelColorProperty()
    {
        if (!$this->diagnosisResults) {
            return 'gray';
        }

        return match($this->diagnosisResults['urgency_level'] ?? 'medium') {
            'low' => 'green',
            'medium' => 'yellow',
            'high' => 'orange',
            'emergency' => 'red',
            default => 'gray'
        };
    }

    public function render()
    {
        return view('livewire.ai.a-i-diagnosis-assistant');
    }
}