<?php

namespace App\Console\Commands;

use App\Models\Prescription;
use Illuminate\Console\Command;

class UpdatePrescriptionStatus extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'prescriptions:update-status';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update prescription statuses - mark expired prescriptions as expired';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Checking prescription statuses...');
        
        // Find active prescriptions that have expired
        $expiredPrescriptions = Prescription::where('status', 'active')
            ->where('valid_until', '<', now()->startOfDay())
            ->get();
        
        $expiredCount = $expiredPrescriptions->count();
        
        if ($expiredCount === 0) {
            $this->info('No expired prescriptions found.');
            return 0;
        }
        
        $this->info("Found {$expiredCount} expired prescription(s) to update.");
        $bar = $this->output->createProgressBar($expiredCount);
        $bar->start();
        
        $updated = 0;
        $errors = 0;
        
        foreach ($expiredPrescriptions as $prescription) {
            try {
                // Update status to expired
                $prescription->status = 'expired';
                $prescription->save();
                
                // The syncMedication will be called automatically via model event
                // which will mark corresponding medications as completed
                $updated++;
            } catch (\Exception $e) {
                $errors++;
                $this->newLine();
                $this->error("Error updating prescription {$prescription->prescription_number}: " . $e->getMessage());
            }
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine(2);
        
        $this->info("✓ Successfully updated {$updated} prescription(s) to expired status.");
        if ($errors > 0) {
            $this->warn("✗ {$errors} prescription(s) had errors.");
        }
        
        // Also check for dispensed prescriptions that might need status updates
        // (This could be expanded based on business logic)
        
        return 0;
    }
}
