<?php

namespace App\Console\Commands;

use App\Models\Prescription;
use Illuminate\Console\Command;

class SyncPrescriptionsToMedications extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'prescriptions:sync-medications {--patient-id= : Sync for specific patient ID only}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync all active prescriptions to medications table';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $patientId = $this->option('patient-id');
        
        $query = Prescription::with('drug')
            ->where('status', 'active');
            
        if ($patientId) {
            $query->where('patient_id', $patientId);
            $this->info("Syncing medications for patient ID: {$patientId}");
        } else {
            $this->info("Syncing medications for all active prescriptions...");
        }
        
        $prescriptions = $query->get();
        $total = $prescriptions->count();
        
        if ($total === 0) {
            $this->warn('No active prescriptions found.');
            return 0;
        }
        
        $this->info("Found {$total} active prescriptions to sync.");
        $bar = $this->output->createProgressBar($total);
        $bar->start();
        
        $synced = 0;
        $errors = 0;
        
        foreach ($prescriptions as $prescription) {
            try {
                $prescription->syncMedication();
                $synced++;
            } catch (\Exception $e) {
                $errors++;
                $this->newLine();
                $this->error("Error syncing prescription {$prescription->prescription_number}: " . $e->getMessage());
            }
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine(2);
        
        $this->info("✓ Successfully synced {$synced} prescriptions.");
        if ($errors > 0) {
            $this->warn("✗ {$errors} prescriptions had errors.");
        }
        
        return 0;
    }
}
