<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use App\Models\Patient;
use App\Models\Drug;
use App\Models\MedicalRecord;
use App\Models\Prescription;
use Carbon\Carbon;

class ImportKaggleDataset extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'kaggle:import {dataset} {--limit=100}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import healthcare datasets from Kaggle for testing';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $dataset = $this->argument('dataset');
        $limit = $this->option('limit');

        $this->info("Importing {$dataset} dataset with limit of {$limit} records...");

        switch ($dataset) {
            case 'heart-disease':
                $this->importHeartDiseaseDataset($limit);
                break;
            case 'diabetes':
                $this->importDiabetesDataset($limit);
                break;
            case 'drug-classification':
                $this->importDrugClassificationDataset($limit);
                break;
            case 'medical-costs':
                $this->importMedicalCostsDataset($limit);
                break;
            case 'sample-patients':
                $this->importSamplePatients($limit);
                break;
            default:
                $this->error("Unknown dataset: {$dataset}");
                $this->info("Available datasets: heart-disease, diabetes, drug-classification, medical-costs, sample-patients");
                return Command::FAILURE;
        }

        $this->info("Dataset import completed successfully!");
        return Command::SUCCESS;
    }

    /**
     * Import sample patients for testing
     */
    private function importSamplePatients($limit)
    {
        $this->info("Creating sample patients...");

        // Get the first user as doctor
        $doctor = User::first();
        if (!$doctor) {
            $this->error("No users found. Please create a user first.");
            return;
        }

        $samplePatients = [
            [
                'user_id' => $doctor->id,
                'patient_id' => 'P' . str_pad(rand(100, 999), 6, '0', STR_PAD_LEFT),
                'first_name' => 'John',
                'last_name' => 'Smith',
                'date_of_birth' => '1985-03-15',
                'gender' => 'male',
                'phone' => '+1-555-0123',
                'address' => '123 Main St, City, State 12345',
                'emergency_contact_name' => 'Jane Smith',
                'emergency_contact_phone' => '+1-555-0124',
                'insurance_provider' => 'Blue Cross Blue Shield',
                'insurance_number' => 'BC123456789',
                'medical_history' => 'Hypertension, Diabetes Type 2',
                'allergies' => 'Penicillin, Shellfish',
                'blood_type' => 'A+',
                'height' => 178.0,
                'weight' => 81.6,
            ],
            [
                'first_name' => 'Sarah',
                'last_name' => 'Johnson',
                'date_of_birth' => '1990-07-22',
                'gender' => 'female',
                'phone' => '+1-555-0125',
                'email' => 'sarah.johnson@email.com',
                'address' => '456 Oak Ave, City, State 12345',
                'emergency_contact' => 'Mike Johnson - +1-555-0126',
                'medical_history' => 'Asthma, Seasonal Allergies',
                'allergies' => 'Pollen, Dust',
                'blood_type' => 'O-',
            ],
            [
                'first_name' => 'Michael',
                'last_name' => 'Brown',
                'date_of_birth' => '1978-11-08',
                'gender' => 'male',
                'phone' => '+1-555-0127',
                'email' => 'michael.brown@email.com',
                'address' => '789 Pine St, City, State 12345',
                'emergency_contact' => 'Lisa Brown - +1-555-0128',
                'medical_history' => 'Heart Disease, High Cholesterol',
                'allergies' => 'None known',
                'blood_type' => 'B+',
            ],
            [
                'first_name' => 'Emily',
                'last_name' => 'Davis',
                'date_of_birth' => '1995-01-14',
                'gender' => 'female',
                'phone' => '+1-555-0129',
                'email' => 'emily.davis@email.com',
                'address' => '321 Elm St, City, State 12345',
                'emergency_contact' => 'Robert Davis - +1-555-0130',
                'medical_history' => 'Migraine, Anxiety',
                'allergies' => 'Latex, Nuts',
                'blood_type' => 'AB+',
            ],
            [
                'first_name' => 'David',
                'last_name' => 'Wilson',
                'date_of_birth' => '1982-09-30',
                'gender' => 'male',
                'phone' => '+1-555-0131',
                'email' => 'david.wilson@email.com',
                'address' => '654 Maple Dr, City, State 12345',
                'emergency_contact' => 'Susan Wilson - +1-555-0132',
                'medical_history' => 'Arthritis, Depression',
                'allergies' => 'Sulfa drugs',
                'blood_type' => 'A-',
            ],
        ];

        $imported = 0;
        foreach ($samplePatients as $patientData) {
            if ($imported >= $limit) break;

            Patient::create($patientData);
            $imported++;
            $this->line("Created patient: {$patientData['first_name']} {$patientData['last_name']}");
        }

        $this->info("Successfully imported {$imported} sample patients.");
    }

    /**
     * Import heart disease dataset
     */
    private function importHeartDiseaseDataset($limit)
    {
        $this->info("Importing heart disease dataset...");
        
        // Sample heart disease data structure
        $heartDiseaseData = [
            [
                'first_name' => 'Robert',
                'last_name' => 'Anderson',
                'date_of_birth' => '1960-05-20',
                'gender' => 'male',
                'phone' => '+1-555-0201',
                'email' => 'robert.anderson@email.com',
                'address' => '100 Cardiac Lane, Heart City, State 12345',
                'medical_history' => 'Heart Disease, Hypertension, High Cholesterol',
                'allergies' => 'None known',
                'blood_type' => 'O+',
                'vital_signs' => [
                    'blood_pressure' => '140/90',
                    'heart_rate' => '85',
                    'temperature' => '98.6°F',
                    'weight' => '180 lbs',
                    'height' => '5\'10"'
                ]
            ],
            [
                'first_name' => 'Maria',
                'last_name' => 'Garcia',
                'date_of_birth' => '1955-08-12',
                'gender' => 'female',
                'phone' => '+1-555-0202',
                'email' => 'maria.garcia@email.com',
                'address' => '200 Pulse Ave, Heart City, State 12345',
                'medical_history' => 'Coronary Artery Disease, Diabetes',
                'allergies' => 'Aspirin',
                'blood_type' => 'A+',
                'vital_signs' => [
                    'blood_pressure' => '160/95',
                    'heart_rate' => '92',
                    'temperature' => '98.4°F',
                    'weight' => '165 lbs',
                    'height' => '5\'6"'
                ]
            ]
        ];

        $imported = 0;
        foreach ($heartDiseaseData as $patientData) {
            if ($imported >= $limit) break;

            $vitalSigns = $patientData['vital_signs'] ?? null;
            unset($patientData['vital_signs']);

            $patient = Patient::create($patientData);
            
            // Create medical record with heart disease information
            if ($vitalSigns) {
                MedicalRecord::create([
                    'patient_id' => $patient->id,
                    'doctor_id' => 1, // Assuming user ID 1 is a doctor
                    'visit_date' => now()->subDays(rand(1, 30)),
                    'chief_complaint' => 'Chest pain and shortness of breath',
                    'diagnosis' => 'Heart Disease',
                    'treatment_plan' => 'Medication management and lifestyle changes',
                    'notes' => 'Patient with cardiovascular risk factors',
                    'vital_signs' => $vitalSigns,
                ]);
            }

            $imported++;
            $this->line("Created heart disease patient: {$patientData['first_name']} {$patientData['last_name']}");
        }

        $this->info("Successfully imported {$imported} heart disease patients.");
    }

    /**
     * Import diabetes dataset
     */
    private function importDiabetesDataset($limit)
    {
        $this->info("Importing diabetes dataset...");
        
        $diabetesData = [
            [
                'first_name' => 'Jennifer',
                'last_name' => 'Taylor',
                'date_of_birth' => '1970-12-03',
                'gender' => 'female',
                'phone' => '+1-555-0301',
                'email' => 'jennifer.taylor@email.com',
                'address' => '300 Sugar St, Diabetes City, State 12345',
                'medical_history' => 'Type 2 Diabetes, Obesity',
                'allergies' => 'None known',
                'blood_type' => 'B+',
                'glucose_levels' => '180 mg/dL',
                'hba1c' => '8.5%'
            ],
            [
                'first_name' => 'James',
                'last_name' => 'Miller',
                'date_of_birth' => '1965-04-18',
                'gender' => 'male',
                'phone' => '+1-555-0302',
                'email' => 'james.miller@email.com',
                'address' => '400 Insulin Ave, Diabetes City, State 12345',
                'medical_history' => 'Type 1 Diabetes, Hypertension',
                'allergies' => 'None known',
                'blood_type' => 'A+',
                'glucose_levels' => '250 mg/dL',
                'hba1c' => '9.2%'
            ]
        ];

        $imported = 0;
        foreach ($diabetesData as $patientData) {
            if ($imported >= $limit) break;

            $glucoseLevels = $patientData['glucose_levels'] ?? null;
            $hba1c = $patientData['hba1c'] ?? null;
            unset($patientData['glucose_levels'], $patientData['hba1c']);

            $patient = Patient::create($patientData);
            
            // Create medical record with diabetes information
            MedicalRecord::create([
                'patient_id' => $patient->id,
                'doctor_id' => 1,
                'visit_date' => now()->subDays(rand(1, 30)),
                'chief_complaint' => 'Blood sugar management',
                'diagnosis' => 'Diabetes Mellitus',
                'treatment_plan' => 'Insulin therapy and diet management',
                'notes' => "Glucose levels: {$glucoseLevels}, HbA1c: {$hba1c}",
                'vital_signs' => [
                    'glucose_level' => $glucoseLevels,
                    'hba1c' => $hba1c,
                    'blood_pressure' => '130/80',
                    'weight' => '200 lbs'
                ],
            ]);

            $imported++;
            $this->line("Created diabetes patient: {$patientData['first_name']} {$patientData['last_name']}");
        }

        $this->info("Successfully imported {$imported} diabetes patients.");
    }

    /**
     * Import drug classification dataset
     */
    private function importDrugClassificationDataset($limit)
    {
        $this->info("Importing drug classification dataset...");
        
        $drugs = [
            [
                'name' => 'Aspirin',
                'generic_name' => 'Acetylsalicylic Acid',
                'manufacturer' => 'Bayer',
                'category' => 'NSAID',
                'description' => 'Nonsteroidal anti-inflammatory drug used for pain relief and blood thinning',
                'dosage_form' => 'Tablet',
                'strength' => '81mg',
                'unit' => 'mg',
                'prescription_required' => false,
                'side_effects' => ['Stomach irritation', 'Bleeding risk', 'Allergic reactions'],
                'contraindications' => ['Bleeding disorders', 'Stomach ulcers', 'Allergy to aspirin'],
                'interactions' => ['Warfarin', 'Ibuprofen', 'Alcohol'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ],
            [
                'name' => 'Metformin',
                'generic_name' => 'Metformin Hydrochloride',
                'manufacturer' => 'Various',
                'category' => 'Antidiabetic',
                'description' => 'First-line medication for type 2 diabetes',
                'dosage_form' => 'Tablet',
                'strength' => '500mg',
                'unit' => 'mg',
                'prescription_required' => true,
                'side_effects' => ['Nausea', 'Diarrhea', 'Stomach upset'],
                'contraindications' => ['Kidney disease', 'Liver disease', 'Heart failure'],
                'interactions' => ['Alcohol', 'Contrast dye', 'Furosemide'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ],
            [
                'name' => 'Lisinopril',
                'generic_name' => 'Lisinopril',
                'manufacturer' => 'Various',
                'category' => 'ACE Inhibitor',
                'description' => 'Angiotensin-converting enzyme inhibitor for hypertension and heart failure',
                'dosage_form' => 'Tablet',
                'strength' => '10mg',
                'unit' => 'mg',
                'prescription_required' => true,
                'side_effects' => ['Dry cough', 'Dizziness', 'High potassium'],
                'contraindications' => ['Pregnancy', 'Kidney disease', 'High potassium'],
                'interactions' => ['Potassium supplements', 'NSAIDs', 'Lithium'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ],
            [
                'name' => 'Atorvastatin',
                'generic_name' => 'Atorvastatin Calcium',
                'manufacturer' => 'Pfizer',
                'category' => 'Statin',
                'description' => 'HMG-CoA reductase inhibitor for cholesterol management',
                'dosage_form' => 'Tablet',
                'strength' => '20mg',
                'unit' => 'mg',
                'prescription_required' => true,
                'side_effects' => ['Muscle pain', 'Liver problems', 'Memory issues'],
                'contraindications' => ['Liver disease', 'Pregnancy', 'Active liver disease'],
                'interactions' => ['Grapefruit juice', 'Warfarin', 'Digoxin'],
                'storage_conditions' => 'Store at room temperature',
                'expiry_alert_days' => 30,
                'is_active' => true,
            ],
            [
                'name' => 'Insulin Glargine',
                'generic_name' => 'Insulin Glargine',
                'manufacturer' => 'Sanofi',
                'category' => 'Insulin',
                'description' => 'Long-acting insulin for diabetes management',
                'dosage_form' => 'Injection',
                'strength' => '100 units/mL',
                'unit' => 'units',
                'prescription_required' => true,
                'side_effects' => ['Low blood sugar', 'Weight gain', 'Injection site reactions'],
                'contraindications' => ['Low blood sugar', 'Allergy to insulin'],
                'interactions' => ['Alcohol', 'Beta-blockers', 'Thiazides'],
                'storage_conditions' => 'Refrigerate, do not freeze',
                'expiry_alert_days' => 7,
                'is_active' => true,
            ]
        ];

        $imported = 0;
        foreach ($drugs as $drugData) {
            if ($imported >= $limit) break;

            Drug::create($drugData);
            $imported++;
            $this->line("Created drug: {$drugData['name']}");
        }

        $this->info("Successfully imported {$imported} drugs.");
    }

    /**
     * Import medical costs dataset
     */
    private function importMedicalCostsDataset($limit)
    {
        $this->info("Importing medical costs dataset...");
        
        $medicalCostsData = [
            [
                'first_name' => 'William',
                'last_name' => 'Jones',
                'date_of_birth' => '1988-06-25',
                'gender' => 'male',
                'phone' => '+1-555-0401',
                'email' => 'william.jones@email.com',
                'address' => '500 Cost Ave, Medical City, State 12345',
                'medical_history' => 'Appendectomy, Broken arm',
                'allergies' => 'Latex',
                'blood_type' => 'O+',
                'insurance_provider' => 'Blue Cross Blue Shield',
                'last_visit_cost' => '$2,500'
            ],
            [
                'first_name' => 'Lisa',
                'last_name' => 'Garcia',
                'date_of_birth' => '1992-09-14',
                'gender' => 'female',
                'phone' => '+1-555-0402',
                'email' => 'lisa.garcia@email.com',
                'address' => '600 Premium St, Medical City, State 12345',
                'medical_history' => 'Regular checkups, Vaccinations',
                'allergies' => 'None known',
                'blood_type' => 'A-',
                'insurance_provider' => 'Aetna',
                'last_visit_cost' => '$150'
            ]
        ];

        $imported = 0;
        foreach ($medicalCostsData as $patientData) {
            if ($imported >= $limit) break;

            $insuranceProvider = $patientData['insurance_provider'] ?? null;
            $lastVisitCost = $patientData['last_visit_cost'] ?? null;
            unset($patientData['insurance_provider'], $patientData['last_visit_cost']);

            $patient = Patient::create($patientData);
            
            // Create medical record with cost information
            MedicalRecord::create([
                'patient_id' => $patient->id,
                'doctor_id' => 1,
                'visit_date' => now()->subDays(rand(1, 30)),
                'chief_complaint' => 'Routine checkup',
                'diagnosis' => 'Healthy',
                'treatment_plan' => 'Continue current lifestyle',
                'notes' => "Insurance: {$insuranceProvider}, Visit cost: {$lastVisitCost}",
            ]);

            $imported++;
            $this->line("Created medical costs patient: {$patientData['first_name']} {$patientData['last_name']}");
        }

        $this->info("Successfully imported {$imported} medical costs patients.");
    }
}
