<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;

class CheckAIConfiguration extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'ai:check-config';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check AI services configuration and connectivity';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Checking AI Configuration...');
        $this->newLine();

        // Check OpenAI API Key
        $this->checkOpenAIConfiguration();

        // Check AI Services
        $this->checkAIServices();

        // Check Database Tables
        $this->checkDatabaseTables();

        $this->newLine();
        $this->info('✅ AI Configuration check completed!');
    }

    private function checkOpenAIConfiguration()
    {
        $this->info('📡 Checking OpenAI Configuration...');

        $apiKey = config('services.openai.api_key');
        
        if (empty($apiKey)) {
            $this->error('❌ OpenAI API key is not configured');
            $this->warn('   Please set OPENAI_API_KEY in your .env file');
            return;
        }

        if (strlen($apiKey) < 20) {
            $this->error('❌ OpenAI API key appears to be invalid (too short)');
            return;
        }

        $this->info('✅ OpenAI API key is configured');

        // Test API connectivity
        try {
            $response = Http::timeout(10)->withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->get('https://api.openai.com/v1/models');

            if ($response->successful()) {
                $this->info('✅ OpenAI API connectivity test passed');
                $models = $response->json()['data'] ?? [];
                $this->info('   Available models: ' . count($models));
            } else {
                $this->error('❌ OpenAI API connectivity test failed');
                $this->warn('   Status: ' . $response->status());
                $this->warn('   Response: ' . $response->body());
            }
        } catch (\Exception $e) {
            $this->error('❌ OpenAI API connectivity test failed: ' . $e->getMessage());
        }

        $this->newLine();
    }

    private function checkAIServices()
    {
        $this->info('🤖 Checking AI Services...');

        $services = [
            'AIDrugInteractionService' => \App\Services\AIDrugInteractionService::class,
            'AIDiagnosisService' => \App\Services\AIDiagnosisService::class,
            'AIPredictiveAnalyticsService' => \App\Services\AIPredictiveAnalyticsService::class,
            'AIChatbotService' => \App\Services\AIChatbotService::class,
        ];

        foreach ($services as $name => $class) {
            if (class_exists($class)) {
                $this->info("✅ {$name} is available");
            } else {
                $this->error("❌ {$name} is not found");
            }
        }

        $this->newLine();
    }

    private function checkDatabaseTables()
    {
        $this->info('🗄️ Checking Database Tables...');

        $tables = [
            'drugs' => 'Drug information for AI analysis',
            'prescriptions' => 'Prescription data for interaction checking',
            'patients' => 'Patient data for AI features',
            'medical_records' => 'Medical history for diagnosis assistance',
        ];

        foreach ($tables as $table => $description) {
            try {
                $exists = \Schema::hasTable($table);
                if ($exists) {
                    $count = \DB::table($table)->count();
                    $this->info("✅ Table '{$table}' exists ({$count} records)");
                } else {
                    $this->error("❌ Table '{$table}' does not exist");
                    $this->warn("   {$description}");
                }
            } catch (\Exception $e) {
                $this->error("❌ Error checking table '{$table}': " . $e->getMessage());
            }
        }

        $this->newLine();
    }
}