# Navigation Menu Grouping Implementation

This document shows how to implement grouped navigation menus with dropdown categories.

## Structure Overview

```
Navigation Menu:
├── Dashboard (single link)
├── Patient Care (dropdown)
│   ├── Patients
│   ├── Visits
│   └── Admissions
├── Clinical Services (dropdown)
│   ├── Consultations
│   ├── Nursing Station
│   ├── Medication Log
│   ├── Lab
│   └── Surgeries
├── Appointments (single link)
├── Pharmacy & Billing (dropdown)
│   ├── Prescriptions
│   └── Billing
└── Administration (dropdown)
    ├── Services
    └── AI Features
```

## Implementation Methods

### Method 1: Using NavigationService (Recommended)

A service class centralizes menu logic and makes it reusable.

**File**: `app/Services/NavigationService.php`
- Contains `getMenuItems()` method that returns structured menu data
- Handles role-based permissions
- Returns array with 'type' => 'link' or 'type' => 'dropdown'

**Usage in Blade**:
```blade
@php
    use App\Services\NavigationService;
    $menuItems = NavigationService::getMenuItems();
@endphp

@foreach($menuItems as $item)
    @if($item['type'] === 'link')
        <x-nav-link ...>{{ $item['label'] }}</x-nav-link>
    @elseif($item['type'] === 'dropdown')
        <x-dropdown>
            <x-slot name="trigger">{{ $item['label'] }}</x-slot>
            <x-slot name="content">
                @foreach($item['items'] as $subItem)
                    <x-dropdown-link :href="route($subItem['route'])">
                        {{ $subItem['label'] }}
                    </x-dropdown-link>
                @endforeach
            </x-slot>
        </x-dropdown>
    @endif
@endforeach
```

### Method 2: Direct Implementation in Blade

Implement dropdowns directly in the navigation template.

**Example**:
```blade
<!-- Patient Care Dropdown -->
<x-dropdown align="left" width="48">
    <x-slot name="trigger">
        <button class="inline-flex items-center px-3 py-2...">
            Patient Care
            <svg class="ms-2 -me-0.5 h-4 w-4">...</svg>
        </button>
    </x-slot>

    <x-slot name="content">
        @if(auth()->user()->canViewPatients())
            <x-dropdown-link :href="route('patients.index')" wire:navigate>
                Patients
            </x-dropdown-link>
        @endif
        <!-- More items... -->
    </x-slot>
</x-dropdown>
```

## Key Components

### 1. Dropdown Component (`resources/views/components/dropdown.blade.php`)
- Already exists in the codebase
- Uses Alpine.js for interactivity
- Supports alignment (left, right, top)
- Supports custom width

### 2. Dropdown Link Component (`resources/views/components/dropdown-link.blade.php`)
- Already exists in the codebase
- Styles dropdown menu items
- Can be used with `wire:navigate` for Livewire navigation

### 3. Active State Detection
- Use `request()->routeIs('route.pattern.*')` to check active state
- For dropdowns, check if any sub-item is active to highlight the dropdown trigger

## Implementation Steps

1. **Create NavigationService** (if using Method 1)
   - Define menu structure
   - Implement permission checks
   - Return structured array

2. **Update Navigation Blade Template**
   - Replace flat menu items with grouped structure
   - Use dropdown component for categories
   - Maintain responsive menu structure

3. **Test Navigation**
   - Verify dropdowns open/close correctly
   - Check active states
   - Test with different user roles
   - Test responsive behavior

## Example Files

- `app/Services/NavigationService.php` - Service class with menu logic
- `resources/views/examples/grouped-navigation-example.blade.php` - Example implementation

## Benefits of Grouping

1. **Better Organization** - Related menu items grouped together
2. **Cleaner Navigation** - Reduces clutter in the main navigation bar
3. **Scalability** - Easy to add new items to categories
4. **User Experience** - More intuitive navigation structure
5. **Maintainability** - Centralized menu logic (with Service approach)

